#include "importlmmworkbook.h"
#include <qdebug.h>

using namespace Qt::StringLiterals;

importlmmworkbook::importlmmworkbook(QObject *parent)
    : QObject(parent), debugMode(false)
{
}

importlmmworkbook::importlmmworkbook(QString fileName)
    : debugMode(false)
{
    Init(fileName);
}

importlmmworkbook::importlmmworkbook(QString fileName, QString language, bool debugMode)
    : debugMode(debugMode)
{
    forcedLanguage = language;
    Init(fileName);
}

importlmmworkbook::importlmmworkbook(QString fileName, bool useEpubLangCode)
    : debugMode(true)
{
    epb.useEpubLangCode = useEpubLangCode;
    Init(fileName);
}

void importlmmworkbook::Init(QString fileName)
{
    sql = &Singleton<sql_class>::Instance();

    epubSectionOffset = 0;
    epubliIdx = 0;

    int yr(0);
    int mo(0);
    QRegularExpression extractYearAndMonth("mwb_.+(\\d\\d\\d\\d\\d\\d)");
    QRegularExpressionMatch match = extractYearAndMonth.match(fileName);
    if (match.hasMatch()) {
        QString txt = match.captured(1);
        yr = QStringView { txt }.left(4).toInt();
        mo = QStringView { txt }.mid(4, 2).toInt();
        if (yr > 2015 && yr < 3000 && mo > 0 && mo < 13) {
            year = yr;
            month = mo;
        } else {
            yr = 0;
        }
    }
    if (yr == 0) {
        year = -1;
        month = -1;
    }
    firstDate.clear();
    lastDate.clear();
    dtBeingImported = nullDate;
    prepared = !epb.Prepare(fileName);
}

importlmmworkbook::~importlmmworkbook()
{
    qDeleteAll(tocEntriesDate);
}

QString importlmmworkbook::getWtLocale(QString languageCode)
{
    QString wtLocale = "E";
    sql = &Singleton<sql_class>::Instance();
    sql_items l;
    l = sql->selectSql("SELECT * FROM languages WHERE code = '" + languageCode + "'", NULL);
    if (l.empty()) {
        // use language code from the settings if the given code wasn't found
        // maybe due to changes, e.g. Qt locale for "no" is nb_NO
        QString lsetting = sql->getSetting("theocbase_language");
        l = sql->selectSql("SELECT * FROM languages WHERE code = '" + lsetting + "'", NULL);
    }
    if (!l.empty()) {
        wtLocale = l[0].value("wt_locale").toString();
    }
    return wtLocale;
}

void importlmmworkbook::fillDocumentIds(int year)
{
    // get first date of a week (ISO 8601)
    QDate firstThursdayofYear(year, 1, 1);
    if (!firstThursdayofYear.isValid())
        return;
    while (firstThursdayofYear.dayOfWeek() != Qt::Thursday)
        firstThursdayofYear = firstThursdayofYear.addDays(1);
    QDate dt = firstThursdayofYear.addDays(-(Qt::Thursday - Qt::Monday)); // start date of the week (with Jan 1)
    if (dt.year() < year)
        dt = dt.addDays(7); // move to first Monday in January

    int issueStartId(0);
    int previousIssueStartId(0);
    int currentWeekIndex = 0;
    sql_item item;
    item.insert(":year", QVariant(year).toString());
    sql = &Singleton<sql_class>::Instance();
    QDate memorialDate = sql->selectScalar("select MIN(date) from exceptions where strftime('%Y', date) = :year and type = 2 and active", &item).toDate();
    QDate memorialMonthSecondWeekDate = QDate();
    if (memorialDate.isValid()) {
        memorialMonthSecondWeekDate = memorialDate.addDays(-(memorialDate.dayOfWeek() - Qt::Monday)); // move to the week's start (Monday)
        // save the second week date of the month of the current year's Memorial
        memorialMonthSecondWeekDate = memorialMonthSecondWeekDate.addDays((2 - (memorialMonthSecondWeekDate.day() + 6) / 7) * 7);
    }
    while (dt.year() == year) {
        // the last three digits consist of the issue index multiplied by 80,
        // to which the consecutive number of the article is added
        issueStartId = (dt.month() - 1) / 2 * 80;
        if (previousIssueStartId != issueStartId) {
            // reset every two months
            currentWeekIndex = 0;
            previousIssueStartId = issueStartId;
        }
        currentWeekIndex++;

        if (dt == memorialMonthSecondWeekDate) {
            if (memorialDate.dayOfWeek() <= Qt::Friday) {
                // no midweek meeting, display Memorial page
            } else {
                // midweek meeting schedule is available, skip the Memorial page
                currentWeekIndex++;
            }
        }

        QString currentDocId = QString("20%1%2").arg(year).arg(issueStartId + currentWeekIndex, 3, 10, QLatin1Char('0'));
        mwbDocumentIds[dt] = currentDocId;
        dt = dt.addDays(7);
    }
}

QString importlmmworkbook::getDocumentId(QDate date)
{
    QDate weekStartDate = date.addDays(-(date.dayOfWeek() - Qt::Monday)); // start date of the week (Monday)
    if (!mwbDocumentIds.contains(weekStartDate))
        fillDocumentIds(weekStartDate.year());
    return mwbDocumentIds[weekStartDate];
}

QDate importlmmworkbook::getDocumentDate(QString documentId, MeetingType meetingType)
{
    QDate weekStartDate = QDate();
    switch (meetingType) {
    case MeetingType::MidweekMeeting:
        if (documentId.length() == 9) {
            int year = documentId.mid(2, 4).toInt();
            fillDocumentIds(year);
            weekStartDate = mwbDocumentIds.key(documentId);
        }
        break;
    default:
        break;
    }
    return weekStartDate;
}

QString importlmmworkbook::Import()
{
    validDates = 0;
    lastError.clear();
    firstDate.clear();
    lastDate.clear();
    rxLoadState = rxLoadStates::noLoadAttempt;

    sql->startTransaction();
    newSettingValues.clear();

    if (prepared) {
        connect(&epb, &epub::ProcessTOCEntry, this, &importlmmworkbook::ProcessTOCEntry);
        connect(&epb, &epub::ProcessCoverImage, this, &importlmmworkbook::ProcessCoverImage);
        epb.ImportTOC();
        epb.ImportCover();
    }

    if (!lastError.isEmpty() || firstDate.isEmpty()) {
        sql->rollbackTransaction();
    } else {
        sql->commitTransaction();
    }
    disconnect(&epb, &epub::ProcessTOCEntry, this, &importlmmworkbook::ProcessTOCEntry);
    disconnect(&epb, &epub::ProcessCoverImage, this, &importlmmworkbook::ProcessCoverImage);

    return resultText(year);
}

QString importlmmworkbook::importHtml(QString url, QString html)
{
    validDates = 0;
    lastError.clear();
    firstDate.clear();
    lastDate.clear();
    rxLoadState = rxLoadStates::noLoadAttempt;
    QUrl u(url);
    QStringList urlParts = u.path().split(QLatin1Char('/'), Qt::SkipEmptyParts);
    QString langCode = urlParts.first();
    int year = QDate::currentDate().year();
    QString documentId;
    if (u.host() == "wol.jw.org") {
        documentId = urlParts.last();
        year = documentId.mid(2, 4).toInt();
    } else if (u.host() == "www.jw.org") {
        QRegularExpression yearRegex("\\d{4}");
        QRegularExpressionMatch yearMatch = yearRegex.match(url);
        if (yearMatch.hasMatch()) {
            year = yearMatch.captured(0).toInt();
        }
        QRegularExpression docIdRegex("\\sdocId-(\\d+)\\s");
        QRegularExpressionMatch docIdMatch = docIdRegex.match(html);
        if (docIdMatch.hasMatch()) {
            documentId = docIdMatch.captured(1);
        }
    } else {
        return "Invalid URL!";
    }

    sql = &Singleton<sql_class>::Instance();
    // get language from html attribute
    QRegularExpression langRegex("\\slang=['\"](?<lang>[\\w\\-]+)['\"]\\s");
    QRegularExpressionMatch langMatch = langRegex.match(html);
    if (langMatch.hasMatch()) {
        langCode = langMatch.captured("lang");
    }
    // get library language code
    langRegex = QRegularExpression("\\s(data-lang=['\"]|ml-)(?<lang>\\w+)(['\"])*\\s");
    langMatch = langRegex.match(html);
    QString libraryLangCode;
    if (langMatch.hasMatch()) {
        libraryLangCode = langMatch.captured("lang");
    }
    if ((langCode == "en" && libraryLangCode != "E")
        || langCode.compare(libraryLangCode, Qt::CaseInsensitive) == 0) {
        // prefer db value when the lang value appears to be wrong
        QString dbLangCode = sql->getLanguageCode(libraryLangCode);
        langCode = !dbLangCode.isEmpty() ? dbLangCode : langCode;
    }

    validDates = 0;
    epb.curlocal = new QLocale(langCode);
    epb.language = langCode;
    LoadRegex(langCode, libraryLangCode);
    if (!lastError.isEmpty())
        return lastError;

    dtBeingImported = nullDate;

    QRegularExpression regex("(<img.+?>|<hr.+?>|<input.+?>)");
    QRegularExpressionMatchIterator gi = regex.globalMatch(html);
    while (gi.hasNext()) {
        auto match = gi.next();
        if (match.hasMatch()) {
            QString elem = match.captured(0);
            html = html.replace(elem, "");
        }
    }

    // html = "<!DOCTYPE inline_dtd[<!ENTITY nbsp \"&#160;\">]>" + html;
    html = html.replace("&quot;", "\"").replace("&nbsp;", " ");
    htmldata = html;

    epubSectionOffset = 0;
    epubliIdx = 0;

    sql->startTransaction();
    newSettingValues.clear();

    lastError.clear();

    ProcessTOCEntry(documentId, "");

    if (!lastError.isEmpty() || firstDate.isEmpty()) {
        sql->rollbackTransaction();
    } else {
        sql->commitTransaction();
    }

    return resultText(year);
}

QString importlmmworkbook::importFile(const QString fileName)
{
    Init(fileName);
    return Import();
}

void importlmmworkbook::ImportDate(QDate dt, QString href, QString chapter)
{
    dtBeingImported = dt;
    ProcessTOCEntry(href, chapter);
    dtBeingImported.fromJulianDay(nullDate.toJulianDay());
}

void importlmmworkbook::LoadRegex(QString langCode, QString libraryLangCode)
{
    QString languageCodeWtLocale = epb.language_ex;
    if (!(langCode.isEmpty() && libraryLangCode.isEmpty()))
        languageCodeWtLocale = QString("%1|%2").arg(langCode).arg(libraryLangCode);
    if (epb.curlocal != nullptr && rxLoadState == rxLoadStates::noLoadAttempt) {
        rxLoadState = rxLoadStates::attempted;
        qDebug() << "Load regular expressions with the language setting:" << languageCodeWtLocale;
        sql_items rx = sql->selectSql(QString("SELECT * FROM lmm_workbookregex WHERE lang LIKE '%1'").arg(languageCodeWtLocale));
        if (rx.size() == 0) {
            QString dbLanguageCode = sql->getLanguageCode(libraryLangCode.isEmpty() ? epb.epubLangCode : libraryLangCode);
            languageCodeWtLocale = QString("%1|%2")
                                           .arg(dbLanguageCode.isEmpty() ? "%" : dbLanguageCode)
                                           .arg(libraryLangCode.isEmpty() ? epb.epubLangCode : libraryLangCode);
            qDebug() << "Regular expressions not found - try language setting:" << languageCodeWtLocale;
            rx = sql->selectSql(QString("SELECT * FROM lmm_workbookregex WHERE lang LIKE '%1'").arg(languageCodeWtLocale));
            if (rx.size() == 0) {
                languageCodeWtLocale = "";
                qDebug() << "Regular expressions still not found - try language setting:" << langCode;
                rx = sql->selectSql(QString("SELECT * FROM lmm_workbookregex WHERE lang LIKE '%1'").arg(langCode));
            }
        }
        for (unsigned int i = 0; i < rx.size(); i++) {
            sql_item kv = rx[i];
            regexes.insert(kv.value("key").toString(), kv.value("value").toString());
            if (epb.language_ex.isEmpty() && !languageCodeWtLocale.isEmpty())
                epb.language_ex = kv.value("lang").toString();
        }
        if (regexes.size() > 0) {
            rxSong.setPattern(regexes["song"]);
            if (!rxSong.isValid())
                qDebug() << "Invalid regular expression to parse the songs!";
            rxTiming.setPattern(regexes["timing"]);
            if (!rxTiming.isValid())
                qDebug() << "Invalid regular expression to parse the week's start date!";
            rxLoadState = rxLoadStates::loaded;
        }
        if (rxLoadState == rxLoadStates::attempted) {
            lastError = tr("Regular expressions are missing for language '%1'!", "Import schedule").arg(epb.language_ex);
        }
    }
}

void importlmmworkbook::ProcessTOCEntry(QString href, QString chapter)
{
    QDate dt(dtBeingImported);
    if (!dt.isValid()) {
        dt.setDate(year, month, 1);
        QRegularExpression documentIdRegex("^(?<documentId>\\d{9})");
        QRegularExpressionMatch documentIdMatch = documentIdRegex.match(href);
        if (documentIdMatch.hasMatch()) {
            QString documentId = documentIdMatch.captured("documentId");
            // get the date from the document id
            dt = getDocumentDate(documentId, MeetingType::MidweekMeeting);
            if (!dt.isValid())
                return; // return silently from auxiliary epub documents
        }
    }
    if (dt.year() < 2024) {
        lastError = tr("Only current editions of the Life and Ministry Meeting Workbook are supported.");
        return;
    }
    qInfo() << "Import the Life and Ministry Meeting schedule from the week of:" << dt.toString();

    // load regex definitions (if not done yet) to parse the xml reader's results
    LoadRegex();

    // Part 1: parse doc

    epubliIdx = 0;
    epubResults.clear();
    for (int i = 0; i <= foundParts::lastItem; i++) {
        epubResults.append("");
    }

    qDebug() << "";
    qDebug() << dt;
    qDebug() << href;

    xml_reader r(epb.oebpsPath + "/" + href, !htmldata.isEmpty() ? htmldata.toUtf8() : nullptr);
    // xml_reader r(htmldata.isEmpty() ? (epb.oebpsPath + "/" + href) : "", htmldata);

    // The xml_reader pulls tokens one after another and examines their type, name, attributes etc.
    // Here we define the criteria which need to be matched to call the "xmlPartFound"-function,
    // which in turn fills the "epubResults"-list with the individual meeting parts
    r.register_elementsearch("body", xmlPartsContexts::body);
    r.register_attributesearch("div", "class", "itemData", xmlPartsContexts::tocEntry, xmlPartsContexts::body, true);

    r.register_attributesearch("header", "", "", xmlPartsContexts::header); // container of the week's Bible reading program
    r.register_attributesearch("h2", "id", "p2", xmlPartsContexts::bibleReading, xmlPartsContexts::header);
    r.register_elementsearch("strong", xmlPartsContexts::bibleReadingDetail, xmlPartsContexts::bibleReading, true);

    r.register_attributesearch("div", "class", "bodyTxt", xmlPartsContexts::bodyTxt); // container of the meeting schedule
    r.register_elementsearch("h2", xmlPartsContexts::section1, xmlPartsContexts::bodyTxt); // meeting section header
    r.register_elementsearch("h3", xmlPartsContexts::talk, xmlPartsContexts::bodyTxt); // meeting part
    r.register_elementsearch("p", xmlPartsContexts::talk, xmlPartsContexts::bodyTxt); // meeting part timing and sources etc.

    QObject::connect(&r, &xml_reader::found, this, &importlmmworkbook::xmlPartFound);
    ignoreTextDepth = -1;
    ignoreTextContext = -1;

    r.read();

    QRegularExpressionMatch match;

    // cleanup
    int dataCnt = 0;
    QRegularExpression cleanup3("•(\\s+)•", QRegularExpression::UseUnicodePropertiesOption | QRegularExpression::CaseInsensitiveOption);
    for (int rIdx = 0; rIdx <= foundParts::lastItem; rIdx++) {
        epubResults[rIdx] = epubResults[rIdx].trimmed();

        while (epubResults[rIdx].contains("  ")) {
            epubResults[rIdx].replace("  ", " ");
        }
        while (epubResults[rIdx].contains("••")) {
            epubResults[rIdx].replace("••", "");
        }
        while ((match = cleanup3.match(epubResults[rIdx], 0, QRegularExpression::NormalMatch, QRegularExpression::DontCheckSubjectStringMatchOption)).hasMatch()) {
            epubResults[rIdx] =
                    epubResults[rIdx].left(match.capturedStart()) + match.captured(1) + epubResults[rIdx].mid(match.capturedEnd());
        }
        if (epubResults[rIdx].contains("<end>"))
            epubResults[rIdx] = epubResults[rIdx].left(epubResults[rIdx].indexOf("<end>"));

        if (epubResults[rIdx].length() > 0)
            dataCnt++;
    }

    bool validDate = dataCnt > 10 && !epubResults[foundParts::biblereading].isEmpty();

    if (validDate) {
        tocEntries.append("(Date) " + chapter);
    } else {
        tocEntries.append(chapter);
    }
    tocEntriesDate.append(new QDate(dt));
    tocEntriesHTML.append(href);

    if (!validDate) {
        if (debugMode && lastError.isEmpty())
            lastError = "Not a schedule";
        return;
    }

    if (firstDate.isEmpty()) {
        firstDate = QLocale().toString(dt, QLocale::ShortFormat);

        if (!DownloadAssistDoc(dt) || !DownloadAssistDoc(dt.addMonths(1))) {
            if (epb.language != "en") {
                qWarning() << "Meeting data is not available yet. Usually it will be available within a few days.";
            }
        }
    }
    // load assist data
    int talkid_index[15] { 0 };
    getTalkIdIndex(dt, talkid_index);

    lastDate = QLocale().toString(dt.addDays(6), QLocale::ShortFormat);
    validDates++;

    meetingResults.clear();
    meetingResults.append("Date: " + dt.toString(Qt::ISODate));

    // Part 2: add to database

    // prepare regular expressions to get the song numbers, timings and assignments from the "epubResults"-list
    // QRegularExpression rxstudy(regexes["study"], QRegularExpression::UseUnicodePropertiesOption | QRegularExpression::CaseInsensitiveOption);
    QRegularExpression rxstudy("[\\(（]([^\\)]+)[\\)）]", QRegularExpression::UseUnicodePropertiesOption | QRegularExpression::CaseInsensitiveOption);
    QList<QRegularExpression *> rxasgn;
    int assignmentoptions(4); // start with 4 possible regex definitions for assignments
    for (int num = 0; num < assignmentoptions; num++) {
        QString key("assignment" + QString::number(num + 1));
        if (regexes.contains(key)) {
            QRegularExpression *rx = new QRegularExpression(regexes[key], QRegularExpression::UseUnicodePropertiesOption | QRegularExpression::CaseInsensitiveOption);
            rxasgn.append(rx);
            if (!rx->isValid())
                meetingResults.append("invalid " + key + ": " + rx->errorString() + " (offset: " + QString::number(rx->patternErrorOffset()) + ")");
        } else {
            assignmentoptions = num; // set number of assignments as defined in the database
            break;
        }
    }

    // get song numbers
    int bsong(-1);
    int msong(-1);
    int esong(-1);
    int esongPartId(foundParts::lastItem);
    match = rxSong.match(epubResults[foundParts::opensong], 0, QRegularExpression::NormalMatch, QRegularExpression::DontCheckSubjectStringMatchOption);
    if (match.hasMatch()) {
        bsong = epb.stringToInt(match.captured(1));
    } else {
        qWarning() << QString("The first song was not found in '%1' with regex '%2'.").arg(epubResults[foundParts::midsong]).arg(rxSong.pattern());
    }
    match = rxSong.match(epubResults[foundParts::midsong], 0, QRegularExpression::NormalMatch, QRegularExpression::DontCheckSubjectStringMatchOption);
    if (match.hasMatch()) {
        msong = epb.stringToInt(match.captured(1));
    } else {
        qWarning() << QString("The second song was not found in '%1' with regex '%2'.").arg(epubResults[foundParts::midsong]).arg(rxSong.pattern());
    }
    for (int partid = foundParts::lastItem; partid > foundParts::talk1; partid--) {
        if (epubResults[partid].isEmpty())
            continue;
        match = rxSong.match(epubResults[partid], 0, QRegularExpression::NormalMatch, QRegularExpression::DontCheckSubjectStringMatchOption);
        if (match.hasMatch()) {
            // closing song
            esong = epb.stringToInt(match.captured(1));
            esongPartId = partid; // save id of the last found item
            break;
        } else if (partid == foundParts::talk1 + 1) {
            qWarning() << QString("The third song was not found with regex '%1'.").arg(rxSong.pattern());
        }
    }

    // save the date, the Bible reading program and the songs of the meeting
    LMM_Meeting mtg(nullptr);
    mtg.loadMeeting(dt);
    mtg.setBibleReading(epubResults[foundParts::biblereading].replace("•", ""));
    mtg.setSongBeginning(bsong);
    mtg.setSongMiddle(msong);
    mtg.setSongEnd(esong);
    if (debugMode) {
        meetingResults.append("Beginning Song: " + QString::number(bsong));
        meetingResults.append("Middle Song: " + QString::number(msong));
        meetingResults.append("End Song: " + QString::number(esong));
        meetingResults.append("Bible Reading: " + epubResults[foundParts::biblereading]);
    } else
        mtg.save();

    // load talk postfix for the current language
    QString talkPostfixName = "Talk Name Postfix for " + epb.language;
    QString talkPostfix = sql->getSetting(talkPostfixName, ":(");
    if (talkPostfix == ":(") {
        // not saved yet, so check talks and save the talk postifx
        QStringList talks;
        for (int partid = 0; partid <= foundParts::lastItem; partid++) {
            QStringView target(QStringView { epubResults[partid] });
            match = rxTiming.matchView(target, 0, QRegularExpression::NormalMatch, QRegularExpression::DontCheckSubjectStringMatchOption);
            if (match.hasMatch()) {
                QStringView head(target.left(match.capturedStart()));
                QString tail(QStringView { target }.mid(match.capturedEnd()).toString());
                for (int rxselectedidx = 0; rxselectedidx < assignmentoptions; rxselectedidx++) {
                    match = rxasgn[rxselectedidx]->matchView(head, 0, QRegularExpression::NormalMatch, QRegularExpression::DontCheckSubjectStringMatchOption);
                    if (match.hasMatch()) {
                        talks.append(match.captured("theme"));
                    }
                }
            }
        }
        if (talks.count() > 0) {
            QString first = talks.first();
            QString talkPostfix_ = "";
            bool ok = true;
            for (int i = 0; ok && i < 10; i++) {
                talkPostfix = talkPostfix_;
                talkPostfix_ = first.right(i);
                QString t;
                foreach (t, talks) {
                    if (!t.endsWith(talkPostfix_)) {
                        ok = false;
                        break;
                    }
                }
            }
            sql->saveSetting(talkPostfixName, talkPostfix);
            newSettingValues.insert(talkPostfixName, talkPostfix);
        }
    }

    int sequence(0);
    int roworder(0);
    QMap<int, int> talkIdsFound;
    QList<QString> nonMatchedTalks;

    int rowindex = 0;
    int matchedTimingCount = 0;

    // create regex to remove characters such as 'soft hyphen' or 'zero width space', which prevent string comparison
    invisibleCharacters.setPattern("\\p{Cf}");
    // create translated texts to recognize the meeting parts and types
    scheduleTerms.clear();
    scheduleTerms[ScheduleTerm::StartingConversation] = MeetingPartClass::toString(MeetingPart::LMM_FM_StartingConversation, dt).replace(invisibleCharacters, "");
    scheduleTerms[ScheduleTerm::FollowingUp] = MeetingPartClass::toString(MeetingPart::LMM_FM_FollowingUp, dt).replace(invisibleCharacters, "");
    scheduleTerms[ScheduleTerm::MakingDisciples] = MeetingPartClass::toString(MeetingPart::LMM_FM_MakingDisciples, dt).replace(invisibleCharacters, "");
    QString explainingYourBeliefsTitle = MeetingPartClass::toString(MeetingPart::LMM_FM_BeliefsTalk, dt).replace(invisibleCharacters, "");
    explainingYourBeliefsTitle = explainingYourBeliefsTitle.left(explainingYourBeliefsTitle.indexOf("(")).trimmed();
    scheduleTerms[ScheduleTerm::ExplainingYourBeliefs] = explainingYourBeliefsTitle;
    scheduleTerms[ScheduleTerm::Talk] = AssignmentTypeClass::toString(AssignmentType::Talk).replace(invisibleCharacters, "");
    scheduleTerms[ScheduleTerm::Demonstration] = AssignmentTypeClass::toString(AssignmentType::Demonstration).replace(invisibleCharacters, "");
    scheduleTerms[ScheduleTerm::Discussion] = AssignmentTypeClass::toString(AssignmentType::Discussion).replace(invisibleCharacters, "");
    scheduleTerms[ScheduleTerm::Video] = AssignmentTypeClass::toString(AssignmentType::Video).replace(invisibleCharacters, "");

    for (int partid = foundParts::treasures; partid < esongPartId; partid++) {
        if (partid == foundParts::midsong)
            continue; // skip song
        QString target(epubResults[partid]);
        if (target.isEmpty())
            continue; // skip empty texts

        // get meeting parts wich have a valid timing
        match = rxTiming.match(target, 0, QRegularExpression::NormalMatch, QRegularExpression::DontCheckSubjectStringMatchOption);
        if (match.hasMatch()) {
            matchedTimingCount++;
            int timing(epb.stringToInt(match.captured("timing")));
            QString head(target.left(match.capturedStart()));
            QString tail(target.mid(match.capturedEnd()));
            for (int rxselectedidx = 0; rxselectedidx < assignmentoptions; rxselectedidx++) {
                // check each assignment regex to get the theme, the talk id and the sequence
                match = rxasgn[rxselectedidx]->match(head, 0, QRegularExpression::NormalMatch, QRegularExpression::DontCheckSubjectStringMatchOption);
                if (match.hasMatch()) {
                    QString theme = match.captured("theme").replace(invisibleCharacters, "");
                    if (theme.endsWith(talkPostfix))
                        theme = theme.mid(0, theme.length() - talkPostfix.length());

                    QString study("");
                    int idx(talkid_index[rowindex]);
                    int talkID(idx > 0 ? idx : 0);
                    if (talkID < 1 && partid <= foundParts::br) {
                        talkID = partid;
                    } else if (talkID < 1) {
                        qDebug() << "Trying to recognize talk types (based on the current language setting)";
                        if (partid >= foundParts::fm1 && partid <= foundParts::fm6) {
                            if (timing >= 7) {
                                talkID = MeetingPartClass::toTalkId(MeetingPart::LMM_FM_Discussion);
                            } else {
                                if (theme.contains(scheduleTerms[ScheduleTerm::StartingConversation], Qt::CaseInsensitive)) {
                                    talkID = MeetingPartClass::toTalkId(MeetingPart::LMM_FM_StartingConversation);
                                } else if (theme.contains(scheduleTerms[ScheduleTerm::FollowingUp], Qt::CaseInsensitive)) {
                                    talkID = MeetingPartClass::toTalkId(MeetingPart::LMM_FM_FollowingUp);
                                } else if (theme.contains(scheduleTerms[ScheduleTerm::MakingDisciples], Qt::CaseInsensitive)) {
                                    talkID = MeetingPartClass::toTalkId(MeetingPart::LMM_FM_MakingDisciples);
                                } else if (theme.contains(scheduleTerms[ScheduleTerm::ExplainingYourBeliefs], Qt::CaseInsensitive)) {
                                    if (tail.contains(scheduleTerms[ScheduleTerm::Talk], Qt::CaseInsensitive))
                                        talkID = MeetingPartClass::toTalkId(MeetingPart::LMM_FM_BeliefsTalk);
                                    else // if (tail.contains(scheduleTerms[ScheduleTerm::Demonstration], Qt::CaseInsensitive)) // do not check and use 'demonstration' since different expressions are used in some languages
                                        talkID = MeetingPartClass::toTalkId(MeetingPart::LMM_FM_BeliefsDemonstration);
                                } else if (theme.contains(scheduleTerms[ScheduleTerm::Talk], Qt::CaseInsensitive)) {
                                    talkID = MeetingPartClass::toTalkId(MeetingPart::LMM_FM_Talk);
                                }
                            }
                        } else if (partid >= foundParts::talk1 && partid <= foundParts::talk5) {
                            if (timing == 30)
                                talkID = MeetingPartClass::toTalkId(MeetingPart::LMM_CBS);
                            else {
                                if (tail.contains(scheduleTerms[ScheduleTerm::Talk], Qt::CaseInsensitive))
                                    talkID = MeetingPartClass::toTalkId(MeetingPart::LMM_CL_Talk);
                                else if (tail.contains(scheduleTerms[ScheduleTerm::Discussion], Qt::CaseInsensitive))
                                    talkID = MeetingPartClass::toTalkId(MeetingPart::LMM_CL_Discussion);
                                else if (tail.contains(scheduleTerms[ScheduleTerm::Video], Qt::CaseInsensitive))
                                    talkID = MeetingPartClass::toTalkId(MeetingPart::LMM_CL_Video);
                                else
                                    talkID = MeetingPartClass::toTalkId(MeetingPart::LMM_CL_Talk);
                            }
                        }
                    }
                    if (talkID > 0) {
                        if (!talkIdsFound.contains(talkID)) {
                            talkIdsFound.insert(talkID, 0);
                            sequence = 0;
                        } else {
                            talkIdsFound[talkID]++;
                            sequence = talkIdsFound[talkID];
                        }

                        if (talkID >= MeetingPartClass::toTalkId(MeetingPart::LMM_TR_BibleReading) && talkID <= MeetingPartClass::toTalkId(MeetingPart::LMM_FM_Talk) && talkID != MeetingPartClass::toTalkId(MeetingPart::LMM_FM_Discussion)) {
                            QRegularExpressionMatch mStudy = rxstudy.match(tail, 0, QRegularExpression::NormalMatch, QRegularExpression::DontCheckSubjectStringMatchOption);
                            if (mStudy.hasMatch()) {
                                study = mStudy.captured(1);
                                tail = tail.left(mStudy.capturedStart()) + tail.mid(mStudy.capturedEnd());
                            }
                        }
                        // save the meeting part
                        InsertSchedule(talkID, sequence, roworder++, dt, theme, match.captured("source"), tail, timing, study);
                        talkid_index[rowindex] = talkID;
                    } else {
                        if (!nonMatchedTalks.contains(theme))
                            nonMatchedTalks.append(theme);
                        qWarning() << QString("In the week of %1, the type of the talk '%2' could not be recognized.").arg(dt.toString(Qt::ISODate)).arg(theme);
                        // qDebug() << "hasNonMatchedTalk" << partid << theme << match.captured("source") << tail;
                    }
                    rowindex++;
                    break;
                }
            }
        } else {
            qWarning() << QString("The timing was not found in '%1' with regex '%2'.").arg(target).arg(rxTiming.pattern());
        }
    }
    qDeleteAll(rxasgn);

    if (matchedTimingCount == 0)
        lastError = tr("The timing of talks could not be recognized.");
    if (nonMatchedTalks.size() > 0 && lastError.isEmpty())
        lastError = tr("Some talk types could not be recognized.");
    else if (lastError.isEmpty())
        saveTalkIndex(dt, talkid_index);
}

QString importlmmworkbook::readRuby(QXmlStreamReader *xml)
{
    Q_ASSERT(xml->isStartElement() && xml->name() == "ruby"_L1);
    QString innerText;
    while (xml->readNextStartElement()) {
        if (xml->name() == "rb"_L1) {
            innerText += xml->readElementText(QXmlStreamReader::IncludeChildElements);
        } else if (xml->name() == "rt"_L1)
            xml->skipCurrentElement();
        else
            innerText += xml->readElementText();
    }
    return innerText;
}

QString importlmmworkbook::readInnerText(QXmlStreamReader *xml)
{
    QString innerText("");
    bool quitRequest(false);
    while (!xml->atEnd() && !xml->hasError() && !quitRequest) {
        QXmlStreamReader::TokenType tokenType = xml->readNext();
        switch (tokenType) {
        case QXmlStreamReader::TokenType::Characters:
            innerText += xml->text().toString();
            break;
        case QXmlStreamReader::TokenType::StartElement:
            if (xml->name() == "ruby"_L1) {
                innerText += readRuby(xml);
            } else if (xml->name() == "a"_L1) {
                // inner text may contain ruby
                innerText += readInnerText(xml);
            } else
                innerText += xml->readElementText(QXmlStreamReader::IncludeChildElements);
            break;
        case QXmlStreamReader::TokenType::EndElement:
            quitRequest = true;
            break;
        default:
            break;
        }
    }
    return innerText;
}

void importlmmworkbook::xmlPartFound(QXmlStreamReader *xml, QXmlStreamReader::TokenType tokenType, int context, int relativeDepth)
{
    bool grabText = false;

    QStringView name = xml->name();
    // QStringView xmlText = xml->text();
    // QStringView id = xml->attributes().value("id");
    // QMetaEnum contextsMetaEnum = QMetaEnum::fromType<xmlPartsContexts>();
    // QString contextString = contextsMetaEnum.valueToKey(context);
    // qDebug() << "----8<------------------------------------------------";
    // qDebug() << name << "id:" << id << "tokentype: " << xml->tokenString() << " context " << contextString;
    // qDebug() << xmlText;
    // qDebug() << "------------------------------------------------>8----";
    // if (id == "p48")
    //     qDebug() << "id found";

    if (ignoreTextDepth > -1 && ignoreTextContext != context) {
        ignoreTextDepth = -1;
        ignoreTextContext = -1;
    }
    QString par = "";

    switch (context) {
    case xmlPartsContexts::bibleReadingDetail:
        epubSectionOffset = 0;
        epubliIdx = foundParts::biblereading;
        grabText = true;
        break;
    case xmlPartsContexts::section1:
        // n/a
        if (epubSectionOffset < foundParts::treasures)
            epubSectionOffset = foundParts::treasures;
        else if (epubSectionOffset < foundParts::fm1)
            epubSectionOffset = foundParts::fm1;
        else if (epubSectionOffset < foundParts::midsong)
            epubSectionOffset = foundParts::midsong;
        epubliIdx = -1;
        break;
    case xmlPartsContexts::talk: {
        int currentPid = 0;
        switch (tokenType) {
        case QXmlStreamReader::StartDocument:
            if (relativeDepth == 0)
                epubResults[epubSectionOffset + epubliIdx] += "<talk>";
            break;
        case QXmlStreamReader::StartElement: {
            currentPid = xml->attributes().value("data-pid").toInt();
            if (relativeDepth == 0 && name.toString() == "h3") {
                epubliIdx++;
                currentTalkPid = currentPid;
            }
            break;
        }
        default:
            break;
        }
        if (currentTalkPid > 0 && currentPid - currentTalkPid <= 1)
            grabText = true;
        break;
    }
    }

    if (grabText) {
        QString txt;
        if (name.toString() == "h3")
            txt.append("•");
        else if (name.toString() == "li" && tokenType == QXmlStreamReader::StartElement && relativeDepth > 0)
            txt.append("~~");

        if (name.toString() == "a" && xml->attributes().value("href").contains(QStringView(QString("footnote")))) {
            ignoreTextDepth = relativeDepth;
            ignoreTextContext = context;
        }
        if (ignoreTextDepth > -1 && (ignoreTextDepth < relativeDepth || ignoreTextContext != context)) {
            ignoreTextDepth = -1; // we are done ignoring text
            ignoreTextContext = -1;
        }
        if (ignoreTextDepth < 0) {
            QString innerText = readInnerText(xml).trimmed();
            txt.append(innerText.replace("\n", "~~").replace("\r", "~~"));

            if (name.toString() == "h3")
                txt.append("•");
            if (name.toString() == "p") {
                if (epubSectionOffset + epubliIdx < epubResults.size() && !epubResults[epubSectionOffset + epubliIdx].isEmpty())
                    txt.append("<end>");
            }
        }
        if (epubResults.size() <= epubSectionOffset + epubliIdx) {
        } else {
            epubResults[epubSectionOffset + epubliIdx] += txt;
        }
    }
}

void importlmmworkbook::ProcessCoverImage(QString fileName)
{
    // format of the setting: <reference pixel x position>|<reference pixel y position>,<January color>,..,<December color>
    // reference pixel position is used to read the color of the given pixel from the cover image
    QString defaultMonthlyColors(QString("50|188,#,#,#,#,#,#,#,#,#,#,#,#").replace("#", "#656164"));
    QString epubColors(sql->getSetting("mwb_colors", defaultMonthlyColors));
    QStringList colors = epubColors.split(",");
    if (colors.size() != 13)
        colors = defaultMonthlyColors.split(",");
    int x(50);
    int y(188);
    if (colors[0].contains("|")) {
        QStringList pixel = colors[0].split("|");
        x = pixel[0].toInt();
        y = pixel[1].toInt();
    }
    QImage img(fileName);
    QColor coverColor = img.pixelColor(x, y);
    if (month > 0 && month < colors.size() - 1 && coverColor != QColorConstants::White) {
        colors[month] = coverColor.name();
        colors[month + 1] = coverColor.name();
    }
    sql->saveSetting("mwb_colors", colors.join(","));
}

bool importlmmworkbook::DownloadAssistDoc(QDate sampleDate)
{
    QNetworkAccessManager manager;
    QEventLoop q;
    bool imported(false);

    QObject::connect(&manager, SIGNAL(finished(QNetworkReply *)),
                     &q, SLOT(quit()));

    QNetworkRequest request;
    request.setTransferTimeout(10000);
    QString sUrl;
    if (sampleDate.isValid())
        sUrl = "https://www.theocbase.net/workbook/workbook" + sampleDate.toString("yyMM") + ".txt";

    request.setUrl(QUrl(sUrl));
    QNetworkReply *reply = manager.get(request);

    q.exec();

    // download complete
    if (reply->error() != QNetworkReply::NoError) {
        qDebug() << "DownloadAssistDoc error" << reply->errorString();
    } else {
        char ln[20];
        int talkid_index[15] { 0 };
        bool ctinue(true);
        QDate dt;
        while (ctinue) {
            ctinue = reply->readLine(ln, 20) != -1;
            if (ctinue) {
                QString ln2(QString::fromUtf8(ln));
                QStringList parts(ln2.split('\t'));
                if (parts.size() == 3) {
                    QDate dt2 = QDate::fromString(parts[0], "yyyy-MM-dd");
                    if (dt.isValid()) {
                        if (dt != dt2) {
                            saveTalkIndex(dt, talkid_index);
                            for (int i = 0; i < 15; i++)
                                talkid_index[i] = 0;
                        }
                    }
                    talkid_index[QVariant(parts[1]).toInt()] = QVariant(parts[2]).toInt();
                    dt = dt2;
                }
            } else if (dt.isValid()) {
                saveTalkIndex(dt, talkid_index);
                imported = true;
            }
        }
    }

    if (reply != nullptr)
        delete reply;
    return imported;
}

// separate method so we can do some additional cleanup on theme or source
void importlmmworkbook::InsertSchedule(int talkId, int sequence, int roworder, QDate date, QString theme, QString source1, QString source2, int time, QString studyText)
{
    QString source(source1.length() > 3 ? source1.trimmed() : source2.trimmed());
    if (debugMode) {
        meetingResults.append(
                LMM_Schedule::getStringTalkType(talkId) + " (seq " + QString::number(sequence) + "): " + cleanText(theme.trimmed()) + " / min " + QString::number(time) + ", study " + studyText + " / " + cleanText(source));
    } else {
        MeetingPart meetingPart = MeetingPartClass::fromInt(talkId);
        LMM_Schedule sch(meetingPart, sequence, roworder, date, cleanText(theme.trimmed()), cleanText(source), studyText, time);
        sch.save();
    }
}

QString importlmmworkbook::cleanText(QString text)
{
    while (text.contains("~~~")) {
        text = text.replace("~~~", "~~");
    }
    while (text.endsWith("~~")) {
        text = text.left(text.length() - 2);
    }
    text = text.replace("~~", "\r\n");
    while (text.startsWith('.') || text.startsWith("•") || text.startsWith(':') || text.startsWith('\r') || text.startsWith('\n')) {
        text = text.mid(1).trimmed();
    }
    while (text.endsWith(':') || text.endsWith("•") || text.endsWith('\r') || text.endsWith('\n')) {
        text = text.left(text.length() - 1).trimmed();
    }
    int space(-1);
    while ((space = text.indexOf("  ")) > -1) {
        text = text.left(space - 1) + text.mid(space + 1);
    }

    // remove enumeration
    text = text.remove(QRegularExpression("^\\p{Nd}{1,2}(\\.\\p{Nd}{1,3})?.")).trimmed();

    return text;
}

void importlmmworkbook::getTalkIdIndex(QDate dt, int (&talkid_index)[15])
{
    sql_item thisdate;
    thisdate.insert(":date", dt);
    sql_items temp = sql->selectSql("select position, talk_id from lmm_schedule_assist where date = :date", &thisdate);
    for (sql_item v : temp) {
        bool ok(false);
        int position(v["position"].toInt(&ok));
        if (ok && position < 15) {
            talkid_index[position] = v["talk_id"].toInt();
        }
    }
}

void importlmmworkbook::saveTalkIndex(QDate dt, int (&talkid_index)[15])
{
    for (int i = 0; i < 15; i++) {
        if (talkid_index[i]) {
            sql_item data;
            data.insert(":date", dt);
            data.insert(":position", i);
            data.insert(":talk_id", talkid_index[i]);
            sql->execSql("insert or replace into lmm_schedule_assist(date, position, talk_id) values(:date, :position, :talk_id)", &data);
        }
    }
}

QString importlmmworkbook::resultText(int year)
{
    if (!lastError.isEmpty()) {
        return lastError;
    } else if (firstDate.isEmpty() || validDates < 1) {
        if (epb.language.isEmpty())
            return QObject::tr("Unable to read new Workbook format");
        if (regexes.isEmpty())
            return QObject::tr("Database not set to handle language '%1'").arg(epb.language);
        if (year < 1)
            return QObject::tr("Unable to find year/month");
        return QObject::tr("Nothing imported (no dates recognized)");
    } else if (debugMode) {
        return ""; // no news is good news
    } else {
        if (validDates == 1) {
            return QObject::tr("Imported week of %1", "Import schedule").arg(firstDate);
        } else
            return QObject::tr("Imported %1 week(s) from %2 thru %3", "Import schedule", validDates).arg(QVariant(validDates).toString(), firstDate, lastDate);
    }
}
