/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2015, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "generatexml.h"
#include <QMap>
#include <QFile>
#include <QMessageBox>
#include <QXmlStreamWriter>
#include <QXmlStreamReader>
#include <QMapIterator>

generateXML::generateXML()
{
    sql = &Singleton<sql_class>::Instance();
}
// code for destructor
//   GenerateXML::~GenerateXML()
//   {

//   }

void generateXML::writeInfo(QXmlStreamWriter *xWriter, int version)
{
    ccongregation c;
    xWriter->writeStartElement("info");
    xWriter->writeTextElement("version", QVariant(version).toString());
    ccongregation::congregation mine = c.getMyCongregation();
    xWriter->writeTextElement("default_congregation", mine.name);

    ccongregation::meeting_dayandtime const &daytime_now = mine.getPublicmeeting_now();
    QString yr = QString::number(daytime_now.getOfyear());
    xWriter->writeTextElement("meeting_day_" + yr, QVariant(daytime_now.getMeetingday()).toString());
    xWriter->writeTextElement("meeting_time_" + yr, daytime_now.getMeetingtime());

    auto const &daytime = mine.getPublicmeeting_next();
    yr = QString::number(daytime.getOfyear());
    xWriter->writeTextElement("meeting_day_" + yr, QVariant(daytime.getMeetingday()).toString());
    xWriter->writeTextElement("meeting_time_" + yr, daytime.getMeetingtime());

    xWriter->writeEndElement(); // info
}

void generateXML::writePersons(QXmlStreamWriter *xWriter, bool publishers, bool speakers)
{
    cpersons *cp = new cpersons();
    int types = 0;
    if (publishers && speakers) {
        types = 2;
    } else if (speakers) {
        types = 1;
    }
    QList<Person *> allusers = cp->getAllPersons(types);
    foreach (Person *p, allusers) {
        xWriter->writeStartElement("person"); // <Person>
        xWriter->writeTextElement("firstname", p->firstName());
        xWriter->writeTextElement("lastname", p->lastName());
        xWriter->writeTextElement("usefor", QString::number(p->usefor()));
        xWriter->writeTextElement("congregation", p->congregationName());
        xWriter->writeTextElement("email", p->email());
        xWriter->writeTextElement("phone", p->phone());
        xWriter->writeTextElement("gender", p->gender() == Person::Male ? "B" : "S");
        xWriter->writeTextElement("servant", QVariant(p->servant()).toString());
        xWriter->writeTextElement("uuid", p->uuid());
        xWriter->writeEndElement(); // </person>
    }
}

void generateXML::writeFamilies(QXmlStreamWriter *xWriter)
{
    QList<family *> flist = family::getFamilies();
    foreach (family *f, flist) {
        Person *familyHead = cpersons::getPerson(f->getHeadId());
        foreach (Person *p, f->getMembers()) {
            if (!p)
                continue;
            xWriter->writeStartElement("family");
            xWriter->writeTextElement("head_firstname", familyHead ? familyHead->firstName() : "");
            xWriter->writeTextElement("head_lastname", familyHead ? familyHead->lastName() : "");
            xWriter->writeTextElement("member_firstname", p->firstName());
            xWriter->writeTextElement("member_lastname", p->lastName());
            xWriter->writeEndElement();
        }
    }
    qDeleteAll(flist);
    flist.clear();
}

void generateXML::writePublicMeetings(QXmlStreamWriter *xWriter, QDate tempdate, int spinbox)
{
    cpersons cp;

    if (sql->getLanguageDefaultId() == 0) {
        QMessageBox::warning(0, "", QObject::tr("Default language not selected!"));
        return;
    }
    cpublictalks c;
    if (spinbox >= 0) {
        for (int weeks = 0; weeks < spinbox + 1; weeks++) {
            QSharedPointer<cptmeeting> cm(c.getMeeting(tempdate));
            if (cm->id > 0) {
                xWriter->writeStartElement("public_meeting");
                xWriter->writeTextElement("date", tempdate.toString(Qt::ISODate));
                xWriter->writeTextElement("theme_id", QVariant(cm->theme.number).toString());
                writeName(xWriter, "speaker", cm->speaker());
                writeName(xWriter, "chairman", cm->chairman());
                writeName(xWriter, "reader", cm->wtReader());
                xWriter->writeTextElement("song_pt", QVariant(cm->songTalk()).toString());
                xWriter->writeTextElement("song_wt_start", QVariant(cm->songWtStart()).toString());
                xWriter->writeTextElement("song_wt_end", QVariant(cm->songWtEnd()).toString());
                xWriter->writeEndElement(); // </public_meeting>
            }
            tempdate = tempdate.addDays(7);
        }
    }

    // all themes
    sql_items esitelmat = sql->selectSql("SELECT * FROM publictalks");
    for (unsigned int j = 0; j < esitelmat.size(); j++) {
        xWriter->writeStartElement("themes");
        xWriter->writeTextElement("theme_id", esitelmat[j].value("theme_number").toString());
        xWriter->writeTextElement("theme", esitelmat[j].value("theme_name").toString());
        xWriter->writeTextElement("lang_id", esitelmat[j].value("lang_id").toString());
        xWriter->writeEndElement();
    }
    sql_items p_esitelmat = sql->selectSql("SELECT speaker_publictalks.*,publictalks.theme_number FROM "
                                           "speaker_publictalks "
                                           "LEFT JOIN publictalks "
                                           "ON speaker_publictalks.theme_id = publictalks.id");
    for (unsigned int k = 0; k < p_esitelmat.size(); k++) {
        xWriter->writeStartElement("speaker_themes");
        Person *speaker = cp.getPerson(p_esitelmat[k].value("speaker_id").toInt());
        if (speaker) {
            xWriter->writeTextElement("speaker_firstname", speaker->firstName());
            xWriter->writeTextElement("speaker_lastname", speaker->lastName());
        } else {
            xWriter->writeTextElement("speaker_firstname", "");
            xWriter->writeTextElement("speaker_lastname", "");
        }
        xWriter->writeTextElement("theme_id", p_esitelmat[k].value("theme_number").toString());
        xWriter->writeTextElement("lang_id", p_esitelmat[k].value("lang_id").toString());
        xWriter->writeEndElement();
    }
}

void generateXML::writeOutgoing(QXmlStreamWriter *xWriter, QDate tempdate, int spinbox)
{
    Q_UNUSED(xWriter);
    Q_UNUSED(tempdate);
    Q_UNUSED(spinbox);
}

void generateXML::writeMidweekMeeting(QXmlStreamWriter *xWriter, QDate tempdate, int spinbox)
{
    LMM_Meeting lmm;
    xWriter->writeTextElement("qty", QVariant(lmm.classes()).toString());

    LMM_Meeting *mtg = new LMM_Meeting();
    for (int weeks = 0; weeks < spinbox + 1; weeks++) {
        if (mtg->loadMeeting(tempdate, false)) {
            xWriter->writeStartElement("midweekmeeting");
            xWriter->writeTextElement("date", mtg->date().toString(Qt::ISODate));
            xWriter->writeTextElement("bible_reading", mtg->bibleReading());
            writeName(xWriter, "chairman", mtg->chairman());
            writeName(xWriter, "counselor2", mtg->counselor2());
            writeName(xWriter, "counselor3", mtg->counselor3());
            writeName(xWriter, "prayer_beginning", mtg->prayerBeginning());
            writeName(xWriter, "prayer_end", mtg->prayerEnd());
            xWriter->writeTextElement("song_beginning", QString::number(mtg->songBeginning()));
            xWriter->writeTextElement("song_middle", QString::number(mtg->songMiddle()));
            xWriter->writeTextElement("song_end", QString::number(mtg->songEnd()));

            QList<LMM_Assignment *> prog = mtg->getAssignments();
            for (LMM_Assignment *asgn : prog) {
                if (asgn->theme() == "")
                    continue;
                xWriter->writeStartElement("talk");
                xWriter->writeTextElement("talk_id", QString::number(asgn->dbTalkId()));
                xWriter->writeTextElement("theme", asgn->theme());
                xWriter->writeTextElement("source", asgn->source().replace("{br}", "\r\n"));
                xWriter->writeTextElement("time", QString::number(asgn->time()));
                xWriter->writeTextElement("class", QString::number(asgn->classnumber()));
                writeName(xWriter, "speaker", asgn->speaker());
                if (asgn->canCounsel()) {
                    LMM_Assignment_ex *studenttalk((LMM_Assignment_ex *)asgn);
                    if (asgn->canHaveAssistant())
                        writeName(xWriter, "assistant", studenttalk->assistant());
                    writeName(xWriter, "volunteer", studenttalk->volunteer());
                    xWriter->writeTextElement("completed", QVariant(studenttalk->completed()).toString());
                } else if (asgn->meetingPart() == MeetingPart::LMM_CBS) {
                    LMM_Assignment_ex *cbs((LMM_Assignment_ex *)asgn);
                    writeName(xWriter, "reader", cbs->assistant());
                }
                xWriter->writeEndElement();
            }

            xWriter->writeEndElement();
        }
        tempdate = tempdate.addDays(7);
    }
    delete mtg;
}

void generateXML::writeName(QXmlStreamWriter *xWriter, QString elementName, Person *person)
{
    if (person) {
        xWriter->writeTextElement(elementName + "_firstname", person->firstName());
        xWriter->writeTextElement(elementName + "_lastname", person->lastName());
    } else {
        xWriter->writeTextElement(elementName + "_firstname", "");
        xWriter->writeTextElement(elementName + "_lastname", "");
    }
}
