#include "dataobject.h"

DataObject::DataObject(const int &id, const QString &name)
    : m_id(id), m_name(name), m_color(""), m_isValid(true)
{
}

DataObject::DataObject(const int &id, const QString &name, const QString &color)
    : m_id(id), m_name(name), m_color(color), m_isValid(true)
{
}

DataObject::DataObject(const int &id, const QString &name, const QString &color, const bool isValid)
    : m_id(id), m_name(name), m_color(color), m_isValid(isValid)
{
}

int DataObject::id() const
{
    return m_id;
}

QString DataObject::name() const
{
    return m_name;
}

QString DataObject::color() const
{
    return m_color;
}

bool DataObject::isValid() const
{
    return m_isValid;
}

DataObjectListModel::DataObjectListModel(QObject *parent)
    : QAbstractListModel(parent)
{
}

QHash<int, QByteArray> DataObjectListModel::roleNames() const
{
    QHash<int, QByteArray> roles;
    roles[IdRole] = "id";
    roles[NameRole] = "name";
    roles[ColorRole] = "color";
    roles[IsValid] = "isValid";
    return roles;
}

void DataObjectListModel::addDataObject(const DataObject &dataObject)
{
    beginInsertRows(QModelIndex(), rowCount(), rowCount());
    m_dataObjects << dataObject;
    endInsertRows();
}

void DataObjectListModel::addDataObject(const int &id, const QString &name, const QString &color, const bool isValid)
{
    DataObject newDataObject(id, name, color, isValid);
    addDataObject(newDataObject);
}

int DataObjectListModel::rowCount(const QModelIndex &parent) const
{
    Q_UNUSED(parent)
    return m_dataObjects.count();
}

QVariant DataObjectListModel::data(const QModelIndex &index, int role) const
{
    int row = index.row();

    if (row < 0 || row > m_dataObjects.count())
        return QVariant();

    const DataObject dataObject = m_dataObjects[index.row()];

    switch (role) {
    case IdRole:
        return dataObject.id();
    case NameRole:
        return dataObject.name();
    case ColorRole:
        return dataObject.color();
    case IsValid:
        return dataObject.isValid();
    }

    return QVariant();
}

int DataObjectListModel::getId(int index) const
{
    if (index < 0 || index > m_dataObjects.count())
        return -1;
    const DataObject dataObject = m_dataObjects[index];
    return dataObject.id();
}

QVariantMap DataObjectListModel::get(int row)
{
    QHash<int, QByteArray> names = roleNames();
    QHashIterator<int, QByteArray> i(names);
    QVariantMap res;
    QModelIndex idx = index(row, 0);
    while (i.hasNext()) {
        i.next();
        QVariant data = idx.data(i.key());
        res[i.value()] = data;
    }
    return res;
}

int DataObjectListModel::getIndex(int id) const
{
    int rows = rowCount();

    for (int i = 0; i < rows; i++) {
        QModelIndex idx = index(i, 0);
        if (id == idx.data(IdRole).toInt()) {
            return i;
        }
    }
    return -1;
}

int DataObjectListModel::getIndex(QString name) const
{
    int rows = rowCount();

    for (int i = 0; i < rows; i++) {
        QModelIndex idx = index(i, 0);
        if (name == idx.data(NameRole).toString()) {
            return i;
        }
    }
    return -1;
}

DataObjectListSortFilterProxyModel::DataObjectListSortFilterProxyModel(QObject *parent)
    : QSortFilterProxyModel(parent)
{
}

QObject *DataObjectListSortFilterProxyModel::source() const
{
    return sourceModel();
}

void DataObjectListSortFilterProxyModel::setSource(QObject *source)
{
    setSourceModel(qobject_cast<QAbstractItemModel *>(source));
    emit sourceChanged();
}

QString DataObjectListSortFilterProxyModel::filterText() const
{
    return m_filterText;
}

void DataObjectListSortFilterProxyModel::setFilterText(QString newValue)
{
    beginFilterChange();
    m_filterText = newValue;
    endFilterChange(QSortFilterProxyModel::Direction::Rows);
    emit filterTextChanged();
}

QByteArray DataObjectListSortFilterProxyModel::sortRole() const
{
    return roleNames().value(QSortFilterProxyModel::sortRole());
}

void DataObjectListSortFilterProxyModel::setSortRole(const QByteArray &role)
{
    QSortFilterProxyModel::setSortRole(roleKey(role));
    emit sortChanged();
}

QByteArray DataObjectListSortFilterProxyModel::groupByRole() const
{
    return m_groupByRole;
}

void DataObjectListSortFilterProxyModel::setGroupByRole(const QByteArray &role)
{
    beginResetModel();
    m_groupByRole = role;
    endResetModel();
    emit groupByChanged();
}

bool DataObjectListSortFilterProxyModel::filterAcceptsRow(int sourceRow,
                                                          const QModelIndex &sourceParent) const
{
    QModelIndex index = sourceModel()->index(sourceRow, 0, sourceParent);
    return sourceModel()->data(index, DataObjectListModel::DataObjectRoles::NameRole).toString().contains(filterText(), Qt::CaseInsensitive);
}

bool DataObjectListSortFilterProxyModel::lessThan(const QModelIndex &left, const QModelIndex &right) const
{
    // compare sections first
    switch (roleKey(groupByRole())) {
    case DataObjectListModel::DataObjectRoles::IdRole: {
        QVariant leftData = sourceModel()->data(left, DataObjectListModel::DataObjectRoles::IdRole);
        QVariant rightData = sourceModel()->data(right, DataObjectListModel::DataObjectRoles::IdRole);
        return leftData.toInt() < rightData.toInt();
        break;
    }
    default:
        break;
    }

    // sort by name within sections
    QVariant val1 = sourceModel()->data(left, DataObjectListModel::DataObjectRoles::NameRole);
    QVariant val2 = sourceModel()->data(right, DataObjectListModel::DataObjectRoles::NameRole);
    int compResult = QString::compare(val1.toString(), val2.toString(), Qt::CaseInsensitive);
    return compResult < 0;
}

int DataObjectListSortFilterProxyModel::roleKey(const QByteArray &role) const
{
    QHash<int, QByteArray> roles = roleNames();
    QHashIterator<int, QByteArray> it(roles);
    while (it.hasNext()) {
        it.next();
        if (it.value() == role)
            return it.key();
    }
    return -1;
}
