﻿/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2015, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "cterritories.h"

GeocodeResult::GeocodeResult(QObject *parent)
    : QObject(parent)
{
}

GeocodeResult::~GeocodeResult()
{
}

void GeocodeResult::setCountry(QString value)
{
    m_country = value;
}

void GeocodeResult::setState(QString value)
{
    m_state = value;
}

void GeocodeResult::setCounty(QString value)
{
    m_county = value;
}

void GeocodeResult::setCity(QString value)
{
    m_city = value;
}

void GeocodeResult::setDistrict(QString value)
{
    m_district = value;
}

void GeocodeResult::setStreet(QString value)
{
    m_street = value;
}

void GeocodeResult::setPostalCode(QString value)
{
    m_postalCode = value;
}

void GeocodeResult::setHouseNumber(QString value)
{
    m_houseNumber = value;
}

QString GeocodeResult::houseNumber()
{
    return m_houseNumber;
}

void GeocodeResult::setLatitude(double value)
{
    m_latitude = value;
}

void GeocodeResult::setLongitude(double value)
{
    m_longitude = value;
}

QString GeocodeResult::wktGeometry()
{
    return "POINT(" + QVariant(m_longitude).toString() + " " + QVariant(m_latitude).toString() + ")";
}

void GeocodeResult::setText(QString value)
{
    m_text = value;
}

GeocodeResultModel::GeocodeResultModel()
{
}

GeocodeResultModel::GeocodeResultModel(QObject *parent)
    : QAbstractTableModel(parent)
{
}

QHash<int, QByteArray> GeocodeResultModel::roleNames() const
{
    QHash<int, QByteArray> items;
    items[CountryRole] = "country";
    items[StateRole] = "state";
    items[CountyRole] = "county";
    items[CityRole] = "city";
    items[DistrictRole] = "district";
    items[StreetRole] = "street";
    items[StreetNumberRole] = "streetNumber";
    items[PostalCodeRole] = "postalCode";
    items[LatitudeRole] = "latitude";
    items[LongitudeRole] = "longitude";
    items[TextRole] = "text";
    return items;
}

// QVariant GeocodeResultModel::headerData(int section, Qt::Orientation orientation, int role) const
//{
//     QStringList header;
//     header << QStringLiteral("") << QStringLiteral("Title") << QStringLiteral("Duration") << QStringLiteral("singer");
//     return header.at(section);
// }

QVariant GeocodeResultModel::headerData(int section, Qt::Orientation orientation, int role) const
{
    if (role == Qt::DisplayRole && orientation == Qt::Horizontal) {
        switch (section) {
        case 0:
            return QString("Name");
        case 1:
            return QString("Time");
        case 2:
            return QString("Result");
        default:
            return QVariant();
        }
    }

    return QVariant();
}

int GeocodeResultModel::rowCount(const QModelIndex & /*parent*/) const
{
    return geocodeResults.count();
}
// QStringList header << QStringLiteral("")<< QStringLiteral("Title")<<QStringLiteral("Duration")<< QStringLiteral("singer");
// int GeocodeResultModel::columnCount(const QModelIndex &parent) const
//{
//      Q_UNUSED(parent);
//    return header.size();
// }
int GeocodeResultModel::columnCount(const QModelIndex & /*parent*/) const
{
    return 11;
}

QVariantMap GeocodeResultModel::get(int row)
{
    QHash<int, QByteArray> names = roleNames();
    QHashIterator<int, QByteArray> i(names);
    QVariantMap res;
    QModelIndex idx = index(row, 0);
    while (i.hasNext()) {
        i.next();
        QVariant data = idx.data(i.key());
        res[i.value()] = data;
    }
    return res;
}

QVariant GeocodeResultModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid())
        return QVariant();

    if (index.row() < 0 || index.row() > geocodeResults.count())
        return QVariant();

    if (role == Qt::DisplayRole || role == Qt::EditRole) {
        switch (index.column()) {
        case 0:
            return geocodeResults[index.row()]->address().country();
        case 1:
            return geocodeResults[index.row()]->address().state();
        case 2:
            return geocodeResults[index.row()]->address().county();
        case 3:
            return geocodeResults[index.row()]->address().city();
        case 4:
            return geocodeResults[index.row()]->address().district();
        case 5:
            return geocodeResults[index.row()]->address().street();
        case 6:
            return geocodeResults[index.row()]->address().streetNumber();
        case 7:
            return geocodeResults[index.row()]->address().postalCode();
        case 8:
            return geocodeResults[index.row()]->coordinate().latitude();
        case 9:
            return geocodeResults[index.row()]->coordinate().longitude();
        case 10:
            return geocodeResults[index.row()]->address().text();
        }
    }

    switch (role) {
    case CountryRole:
        return geocodeResults[index.row()]->address().country();
    case StateRole:
        return geocodeResults[index.row()]->address().state();
    case CountyRole:
        return geocodeResults[index.row()]->address().county();
    case CityRole:
        return geocodeResults[index.row()]->address().city();
    case DistrictRole:
        return geocodeResults[index.row()]->address().district();
    case StreetRole:
        return geocodeResults[index.row()]->address().street();
    case StreetNumberRole:
        return geocodeResults[index.row()]->address().streetNumber();
    case PostalCodeRole:
        return geocodeResults[index.row()]->address().postalCode();
    case LatitudeRole:
        return geocodeResults[index.row()]->coordinate().latitude();
    case LongitudeRole:
        return geocodeResults[index.row()]->coordinate().longitude();
    case TextRole:
        return geocodeResults[index.row()]->address().text();
    }

    return QVariant();
}

// QVariant GeocodeResultModel::data(const QModelIndex &index, int role) const
//{
//     switch (role) {
//     case Qt::DisplayRole:
//         return QString("%1, %2").arg(index.column()).arg(index.row());
//     default:
//         break;
//     }

//    return QVariant();
//}

bool GeocodeResultModel::setData(const QModelIndex &index, const QVariant &value, int role)
{
    QGeoLocation *currGeoLocation = geocodeResults[index.row()];

    if (role == Qt::EditRole) {
        int column = index.column();

        switch (column) {
        case 0:
            currGeoLocation->address().setCountry(value.toString());
            emit dataChanged(this->index(index.row(), 0), this->index(index.row(), 0));
            break;
        case 1:
            currGeoLocation->address().setState(value.toString());
            emit dataChanged(this->index(index.row(), 0), this->index(index.row(), 1));
            break;
        case 2:
            currGeoLocation->address().setCounty(value.toString());
            emit dataChanged(this->index(index.row(), 0), this->index(index.row(), 2));
            break;
        case 3:
            currGeoLocation->address().setCity(value.toString());
            emit dataChanged(this->index(index.row(), 0), this->index(index.row(), 3));
            break;
        case 4:
            currGeoLocation->address().setDistrict(value.toString());
            emit dataChanged(this->index(index.row(), 0), this->index(index.row(), 4));
            break;
        case 5:
            currGeoLocation->address().setStreet(value.toString());
            emit dataChanged(this->index(index.row(), 0), this->index(index.row(), 5));
            break;
        case 6:
            currGeoLocation->address().setStreetNumber(value.toString());
            emit dataChanged(this->index(index.row(), 0), this->index(index.row(), 6));
            break;
        case 7:
            currGeoLocation->address().setPostalCode(value.toString());
            emit dataChanged(this->index(index.row(), 0), this->index(index.row(), 7));
            break;
        case 8:
            currGeoLocation->coordinate().setLatitude(value.toDouble());
            emit dataChanged(this->index(index.row(), 0), this->index(index.row(), 8));
            break;
        case 9:
            currGeoLocation->coordinate().setLongitude(value.toDouble());
            emit dataChanged(this->index(index.row(), 0), this->index(index.row(), 9));
            break;
        case 10:
            currGeoLocation->address().setText(value.toString());
            emit dataChanged(this->index(index.row(), 0), this->index(index.row(), 10));
            break;
        default:
            break;
        }
    } else {
        switch (role) {
        case Roles::CountryRole:
            currGeoLocation->address().setCountry(value.toString());
            emit dataChanged(this->index(index.row(), 0), this->index(index.row(), 0));
            break;
        case Roles::StateRole:
            currGeoLocation->address().setState(value.toString());
            emit dataChanged(this->index(index.row(), 1), this->index(index.row(), 1));
            break;
        case Roles::CountyRole:
            currGeoLocation->address().setCounty(value.toString());
            emit dataChanged(this->index(index.row(), 2), this->index(index.row(), 2));
            break;
        case Roles::CityRole:
            currGeoLocation->address().setCity(value.toString());
            emit dataChanged(this->index(index.row(), 3), this->index(index.row(), 3));
            break;
        case Roles::DistrictRole:
            currGeoLocation->address().setDistrict(value.toString());
            emit dataChanged(this->index(index.row(), 4), this->index(index.row(), 4));
            break;
        case Roles::StreetRole:
            currGeoLocation->address().setStreet(value.toString());
            emit dataChanged(this->index(index.row(), 5), this->index(index.row(), 5));
            break;
        case Roles::StreetNumberRole:
            currGeoLocation->address().setStreetNumber(value.toString());
            emit dataChanged(this->index(index.row(), 6), this->index(index.row(), 6));
            break;
        case Roles::PostalCodeRole:
            currGeoLocation->address().setPostalCode(value.toString());
            emit dataChanged(this->index(index.row(), 7), this->index(index.row(), 7));
            break;
        case Roles::LatitudeRole:
            currGeoLocation->coordinate().setLatitude(value.toDouble());
            emit dataChanged(this->index(index.row(), 8), this->index(index.row(), 8));
            break;
        case Roles::LongitudeRole:
            currGeoLocation->coordinate().setLongitude(value.toDouble());
            emit dataChanged(this->index(index.row(), 9), this->index(index.row(), 9));
            break;
        case Roles::TextRole:
            currGeoLocation->address().setText(value.toString());
            emit dataChanged(this->index(index.row(), 10), this->index(index.row(), 10));
            break;
        default:
            break;
        }
    }

    return true;
}

Qt::ItemFlags GeocodeResultModel::flags(const QModelIndex &index) const
{
    if (!index.isValid())
        return Qt::ItemIsEnabled;
    return Qt::ItemIsEditable | QAbstractTableModel::flags(index);
}

QModelIndex GeocodeResultModel::index(int row, int column, const QModelIndex &parent)
        const
{
    if (hasIndex(row, column, parent)) {
        return createIndex(row, column);
    }
    return QModelIndex();
}

void GeocodeResultModel::addGeocodeResult(QGeoLocation *geocodeResult)
{
    beginInsertRows(QModelIndex(), rowCount(), rowCount());
    QGeoLocation *newGeoLocation = new QGeoLocation();
    newGeoLocation->setAddress(geocodeResult->address());
    newGeoLocation->setBoundingShape(geocodeResult->boundingShape());
    newGeoLocation->setCoordinate(geocodeResult->coordinate());
    newGeoLocation->setExtendedAttributes(geocodeResult->extendedAttributes());
    geocodeResults << newGeoLocation;
    endInsertRows();
}

bool GeocodeResultModel::removeRows(int row, int count, const QModelIndex &parent)
{
    Q_UNUSED(parent)
    if (row < 0 || count < 1 || (row + count) > geocodeResults.size())
        return false;
    beginRemoveRows(QModelIndex(), row, row + count - 1);
    for (int i = 0; i < count; i++) {
        geocodeResults.removeAt(row);
    }
    endRemoveRows();
    return true;
}

QGeoLocation *GeocodeResultModel::getItem(const QModelIndex &index) const
{
    if (index.isValid()) {
        QGeoLocation *item = geocodeResults[index.row()];
        if (item)
            return item;
    }
    return nullptr;
}

GeocodeResultSortFilterProxyModel::GeocodeResultSortFilterProxyModel(QObject *parent)
    : QSortFilterProxyModel(parent)
{
}

QObject *GeocodeResultSortFilterProxyModel::source() const
{
    return sourceModel();
}

void GeocodeResultSortFilterProxyModel::setSource(QObject *source)
{
    setSourceModel(qobject_cast<QAbstractItemModel *>(source));
    emit sourceChanged();
}

QString GeocodeResultSortFilterProxyModel::filterText() const
{
    return m_filterText;
}

void GeocodeResultSortFilterProxyModel::setFilterText(QString newValue)
{
    beginFilterChange();
    m_filterText = newValue;
    endFilterChange(QSortFilterProxyModel::Direction::Rows);
    emit filterTextChanged();
}

QByteArray GeocodeResultSortFilterProxyModel::sortRole() const
{
    return roleNames().value(QSortFilterProxyModel::sortRole());
}

void GeocodeResultSortFilterProxyModel::setSortRole(const QByteArray &newSortRole)
{
    QSortFilterProxyModel::setSortRole(roleKey(newSortRole));
    emit sortChanged();
}

QByteArray GeocodeResultSortFilterProxyModel::groupByRole() const
{
    return m_groupByRole;
}

void GeocodeResultSortFilterProxyModel::setGroupByRole(const QByteArray &newGroupByRole)
{
    if (m_groupByRole == newGroupByRole)
        return;
    beginResetModel();
    m_groupByRole = newGroupByRole;
    endResetModel();
    emit groupByRoleChanged();
    sort(0, sortOrder());
}

bool GeocodeResultSortFilterProxyModel::filterAcceptsRow(int sourceRow,
                                                         const QModelIndex &sourceParent) const
{
    QModelIndex index = sourceModel()->index(sourceRow, 0, sourceParent);
    return sourceModel()->data(index, GeocodeResultModel::Roles::TextRole).toString().contains(filterText(), Qt::CaseInsensitive);
}

bool GeocodeResultSortFilterProxyModel::lessThan(const QModelIndex &left, const QModelIndex &right) const
{
    // compare sections first
    int role = roleKey(groupByRole());
    QVariant val1 = sourceModel()->data(left, role);
    QVariant val2 = sourceModel()->data(right, role);
    int compResult = QString::compare(val1.toString(), val2.toString(), Qt::CaseInsensitive);
    if (compResult != 0)
        return compResult < 0;

    // sort by text within sections
    QVariant leftData = sourceModel()->data(left, GeocodeResultModel::Roles::TextRole);
    QVariant rightData = sourceModel()->data(right, GeocodeResultModel::Roles::TextRole);
    return leftData.toString() < rightData.toString();
}

int GeocodeResultSortFilterProxyModel::roleKey(const QByteArray &role) const
{
    QHash<int, QByteArray> roles = roleNames();
    QHashIterator<int, QByteArray> it(roles);
    while (it.hasNext()) {
        it.next();
        if (it.value() == role)
            return it.key();
    }
    return -1;
}

StreetResult::StreetResult(QObject *parent)
    : QObject(parent), m_isChecked(false)
{
}

StreetResult::~StreetResult()
{
}

bool StreetResult::isChecked() const
{
    return m_isChecked;
}

void StreetResult::setIsChecked(bool newIsChecked)
{
    if (m_isChecked == newIsChecked)
        return;
    m_isChecked = newIsChecked;
    emit isCheckedChanged();
}

QString StreetResult::streetName() const
{
    return m_streetName;
}

void StreetResult::setStreetName(const QString &newStreetName)
{
    if (m_streetName == newStreetName)
        return;
    m_streetName = newStreetName;
    emit streetNameChanged();
}

QString StreetResult::wktGeometry() const
{
    return m_wktGeometry;
}

void StreetResult::setWktGeometry(const QString &newWktGeometry)
{
    if (m_wktGeometry == newWktGeometry)
        return;
    m_wktGeometry = newWktGeometry;
    emit wktGeometryChanged();
}

bool StreetResult::isAlreadyAdded() const
{
    return m_isAlreadyAdded;
}

void StreetResult::setIsAlreadyAdded(bool newIsAlreadyAdded)
{
    if (m_isAlreadyAdded == newIsAlreadyAdded)
        return;
    m_isAlreadyAdded = newIsAlreadyAdded;
    emit isAlreadyAddedChanged();
}

StreetResultModel::StreetResultModel()
{
}

StreetResultModel::StreetResultModel(QObject *parent)
    : QAbstractTableModel(parent)
{
}

QHash<int, QByteArray> StreetResultModel::roleNames() const
{
    QHash<int, QByteArray> items;
    items[IsCheckedRole] = "isChecked";
    items[StreetNameRole] = "streetName";
    items[IsAlreadyAddedRole] = "isAlreadyAdded";
    items[WktGeometryRole] = "wktGeometry";
    return items;
}

int StreetResultModel::rowCount(const QModelIndex & /*parent*/) const
{
    return streetResults.count();
}

int StreetResultModel::columnCount(const QModelIndex & /*parent*/) const
{
    return 8;
}

QVariantMap StreetResultModel::get(int row)
{
    QHash<int, QByteArray> names = roleNames();
    QHashIterator<int, QByteArray> i(names);
    QVariantMap res;
    QModelIndex idx = index(row, 0);
    while (i.hasNext()) {
        i.next();
        QVariant data = idx.data(i.key());
        res[i.value()] = data;
    }
    return res;
}

QVariant StreetResultModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid())
        return QVariant();

    if (index.row() < 0 || index.row() > streetResults.count())
        return QVariant();

    if (role == Qt::DisplayRole || role == Qt::EditRole) {
        switch (index.column()) {
        case 0:
            return streetResults[index.row()]->isChecked();
        case 1:
            return streetResults[index.row()]->streetName();
        case 2:
            return streetResults[index.row()]->isAlreadyAdded();
        case 3:
            return streetResults[index.row()]->wktGeometry();
        }
    }

    switch (role) {
    case IsCheckedRole:
        return streetResults[index.row()]->isChecked();
    case StreetNameRole:
        return streetResults[index.row()]->streetName();
    case IsAlreadyAddedRole:
        return streetResults[index.row()]->isAlreadyAdded();
    case WktGeometryRole:
        return streetResults[index.row()]->wktGeometry();
    }

    return QVariant();
}

bool StreetResultModel::setData(const QModelIndex &index, const QVariant &value, int role)
{
    StreetResult *currStreetResult = streetResults[index.row()];

    if (role == Qt::EditRole) {
        int column = index.column();

        switch (column) {
        case 0:
            currStreetResult->setIsChecked(value.toBool());
            emit dataChanged(this->index(index.row(), 0), this->index(index.row(), 0));
            break;
        case 1:
            currStreetResult->setStreetName(value.toString());
            emit dataChanged(this->index(index.row(), 0), this->index(index.row(), 1));
            break;
        case 2:
            currStreetResult->setIsAlreadyAdded(value.toBool());
            emit dataChanged(this->index(index.row(), 0), this->index(index.row(), 2));
            break;
        case 3:
            currStreetResult->setWktGeometry(value.toString());
            emit dataChanged(this->index(index.row(), 0), this->index(index.row(), 3));
            break;
        default:
            break;
        }
    } else {
        switch (role) {
        case Roles::IsCheckedRole:
            currStreetResult->setIsChecked(value.toBool());
            emit dataChanged(this->index(index.row(), 0), this->index(index.row(), 0));
            break;
        case Roles::StreetNameRole:
            currStreetResult->setStreetName(value.toString());
            emit dataChanged(this->index(index.row(), 1), this->index(index.row(), 1));
            break;
        case Roles::IsAlreadyAddedRole:
            currStreetResult->setIsAlreadyAdded(value.toBool());
            emit dataChanged(this->index(index.row(), 2), this->index(index.row(), 2));
            break;
        case Roles::WktGeometryRole:
            currStreetResult->setWktGeometry(value.toString());
            emit dataChanged(this->index(index.row(), 3), this->index(index.row(), 3));
            break;
        default:
            break;
        }
    }

    return true;
}

Qt::ItemFlags StreetResultModel::flags(const QModelIndex &index) const
{
    if (!index.isValid())
        return Qt::ItemIsEnabled;
    return Qt::ItemIsEditable | QAbstractTableModel::flags(index);
}

QModelIndex StreetResultModel::index(int row, int column, const QModelIndex &parent)
        const
{
    if (hasIndex(row, column, parent)) {
        return createIndex(row, column);
    }
    return QModelIndex();
}

void StreetResultModel::addStreetResult(StreetResult *streetResult)
{
    beginInsertRows(QModelIndex(), rowCount(), rowCount());
    streetResults << streetResult;
    endInsertRows();
}

bool StreetResultModel::removeRows(int row, int count, const QModelIndex &parent)
{
    Q_UNUSED(parent)
    if (row < 0 || count < 1 || (row + count) > streetResults.size())
        return false;
    beginRemoveRows(QModelIndex(), row, row + count - 1);
    for (int i = 0; i < count; i++) {
        streetResults.removeAt(row);
    }
    endRemoveRows();
    return true;
}

StreetResult *StreetResultModel::getItem(const QModelIndex &index) const
{
    if (index.isValid()) {
        StreetResult *item = streetResults[index.row()];
        if (item)
            return item;
    }
    return nullptr;
}

StreetResultSortFilterProxyModel::StreetResultSortFilterProxyModel(QObject *parent)
    : QSortFilterProxyModel(parent)
{
}

QObject *StreetResultSortFilterProxyModel::source() const
{
    return sourceModel();
}

void StreetResultSortFilterProxyModel::setSource(QObject *source)
{
    setSourceModel(qobject_cast<QAbstractItemModel *>(source));
}

QString StreetResultSortFilterProxyModel::filterText() const
{
    return m_filterText;
}

void StreetResultSortFilterProxyModel::setFilterText(QString newValue)
{
    beginFilterChange();
    m_filterText = newValue;
    endFilterChange(QSortFilterProxyModel::Direction::Rows);
    emit filterTextChanged();
}

bool StreetResultSortFilterProxyModel::hideAlreadyAdded() const
{
    return m_hideAlreadyAdded;
}

void StreetResultSortFilterProxyModel::setHideAlreadyAdded(bool newValue)
{
    beginFilterChange();
    m_hideAlreadyAdded = newValue;
    endFilterChange(QSortFilterProxyModel::Direction::Rows);
    emit hideAlreadyAddedChanged();
}

bool StreetResultSortFilterProxyModel::filterAcceptsRow(int sourceRow,
                                                        const QModelIndex &sourceParent) const
{
    QModelIndex indexStreetName = sourceModel()->index(sourceRow, 1, sourceParent);
    QModelIndex indexIsAlreadyAdded = sourceModel()->index(sourceRow, 2, sourceParent);
    return (sourceModel()->data(indexStreetName).toString().contains(filterText(), Qt::CaseInsensitive))
            && (hideAlreadyAdded() ? !sourceModel()->data(indexIsAlreadyAdded).toBool() : true);
}

bool StreetResultSortFilterProxyModel::lessThan(const QModelIndex &left, const QModelIndex &right) const
{
    QVariant leftData = sourceModel()->data(left);
    QVariant rightData = sourceModel()->data(right);

    return leftData.toString() < rightData.toString();
}

int Boundary::territoryId() const
{
    return m_territoryId;
}

void Boundary::setTerritoryId(int value)
{
    m_territoryId = value;
}

QGeoPolygon Boundary::geoPolygon() const
{
    return m_geoPolygon;
}

void Boundary::setGeoPolygon(const QGeoPolygon &newGeoPolygon)
{
    if (m_geoPolygon == newGeoPolygon)
        return;
    m_geoPolygon = newGeoPolygon;
}

QVariantList Boundary::coordinates() const
{
    QVariantList coordinates;
    if (m_geoPolygon.isValid()) {
        foreach (const QGeoCoordinate &coordinate, m_geoPolygon.perimeter())
            coordinates << QVariant::fromValue(coordinate);
    }
    return coordinates;
}

int Street::streetId() const
{
    return m_streetId;
}

void Street::setStreetId(int value)
{
    m_streetId = value;
}

int Street::territoryId() const
{
    return m_territoryId;
}

void Street::setTerritoryId(int value)
{
    m_territoryId = value;
}

QGeoPath Street::path() const
{
    return m_path;
}

void Street::setPath(const QGeoPath &value)
{
    m_path = value;
}

QVariantList Street::coordinates() const
{
    QVariantList coordinates;
    if (m_path.isValid()) {
        foreach (const QGeoCoordinate &coordinate, m_path.path())
            coordinates << QVariant::fromValue(coordinate);
    }
    return coordinates;
}

int Street::streetTypeId() const
{
    return m_streetTypeId;
}

void Street::setStreetTypeId(int value)
{
    m_streetTypeId = value;
}

CSVSchema::CSVSchema(QObject *parent)
    : QObject(parent), m_delimiter(""), m_fields(QList<QString>())
{
}

CSVSchema::~CSVSchema()
{
}

QString CSVSchema::delimiter() const
{
    return m_delimiter;
}

void CSVSchema::setDelimiter(const QString &delimiter)
{
    m_delimiter = delimiter;
}

QList<QString> CSVSchema::fields() const
{
    return m_fields;
}

void CSVSchema::setFields(const QList<QString> &fields)
{
    m_fields = fields;
}

cterritories::cterritories(QObject *parent)
    : QObject(parent)
{
    // default Qt geocoding
    // https://stackoverflow.com/questions/50548492/qgeocodingmanager-gives-no-errors-but-no-results
    // https://www.heise.de/hintergrund/Mapping-Geocoding-und-Routing-mit-Qt-Location-3161424.html?seite=3
    QGeoServiceProvider *geoService = new QGeoServiceProvider("osm");
    geocodingManager = geoService->geocodingManager();
    connect(geocodingManager, SIGNAL(finished(QGeoCodeReply *)),
            SLOT(geocodeReplyFinished(QGeoCodeReply *)));
    // alternative geocoding
    addressRequest = new QNetworkAccessManager();
    addressRequest->setTransferTimeout(QNetworkRequest::DefaultTransferTimeoutConstant);
    connect(addressRequest, SIGNAL(finished(QNetworkReply *)),
            SLOT(addressRequestFinished(QNetworkReply *)));
}

cterritories::~cterritories()
{
}

QVariantMap cterritories::setupGeoServices(int defaultGeoServiceProvider)
{
    sql_class *sql = &Singleton<sql_class>::Instance();
    QVariantMap geoServiceParameters;
    QSettings settings;
    switch (defaultGeoServiceProvider) {
    case 2: {
        QString hereAPIKey = settings.value("geo_service_provider/here_api_key", "").toString();
        if (hereAPIKey.isEmpty()) {
            sql_items apiKeys = sql->selectSql("SELECT value FROM settings WHERE name LIKE 'here_api_key_%' AND value <> '' AND active");
            if (!apiKeys.empty()) {
                int apiKeyCount = apiKeys.size();
                int randomIndex = QRandomGenerator::global()->bounded(0, apiKeyCount);
                hereAPIKey = apiKeys[randomIndex].value("value").toString();
            }
        }
        if (!hereAPIKey.isEmpty())
            geoServiceParameters["here.apiKey"] = hereAPIKey;
        break;
    }
    default:
        QString osmAPIKey = settings.value("geo_service_provider/osm_thf_api_key", "").toString();
        if (osmAPIKey.isEmpty()) {
            sql_items apiKeys = sql->selectSql("SELECT value FROM settings WHERE name LIKE 'osm_thf_api_key_%' AND value <> '' AND active");
            if (!apiKeys.empty()) {
                int apiKeyCount = apiKeys.size();
                int randomIndex = QRandomGenerator::global()->bounded(0, apiKeyCount);
                osmAPIKey = apiKeys[randomIndex].value("value").toString();
            }
        }
        if (!osmAPIKey.isEmpty()) {
            // TODO: Adjust setting OSM API key when Qt provides a solution fo this issue
            // see https://bugreports.qt.io/browse/QTBUG-115742
            // (using osm.mapping.custom.host is not a solution!)
            QString osmResource = ":/osm";
            QString repositoryFolder = QStandardPaths::writableLocation(QStandardPaths::AppDataLocation)
                    + QDir::separator() + "osm_repository";
            QDir().mkpath(repositoryFolder);
            QDirIterator it(osmResource, QDirIterator::Subdirectories);
            while (it.hasNext()) {
                QString srcFilePath = it.next();
                QFileInfo srcFileInfo = QFileInfo(srcFilePath);
                QString dstFilePath = srcFilePath;
                dstFilePath.replace(osmResource, repositoryFolder);

                if (srcFileInfo.isFile()) {
                    QFile srcFile(srcFileInfo.filePath());
                    QString data;
                    if (srcFile.open(QIODevice::ReadOnly)) {
                        data = srcFile.readAll();
                    }
                    srcFile.close();
                    data.replace("YOUR_API_KEY", osmAPIKey);
                    QFile dstFile(dstFilePath);
                    if (dstFile.open(QIODevice::WriteOnly)) {
                        dstFile.write(data.toUtf8());
                    }
                    dstFile.close();
                }
            }
            geoServiceParameters["osm.mapping.providersrepository.address"] = QUrl::fromLocalFile(repositoryFolder); // "file:///home/marlon/QtProjects/theocbase_changes/qt-osm-map-providers/";
            geoServiceParameters["osm.mapping.highdpi_tiles"] = "true";
        }

        // QString osmMappingCustomHost = settings.value("geo_service_provider/osm_mapping_custom_host", "").toString();
        // osmMappingCustomHost = "https://tile.memomaps.de/tilegen/";
        // if (!osmMappingCustomHost.isEmpty()) {
        //     // geoServiceParameters["osm.mapping.providersrepository.disabled"] = "true";
        //     geoServiceParameters["osm.mapping.custom.host"] = osmMappingCustomHost;
        // }

        // QString osmGeocodingHost = settings.value("geo_service_provider/osm_geocoding_host", "").toString();
        // if (!osmGeocodingHost.isEmpty())
        //     geoServiceParameters["osm.geocoding.host"] = osmGeocodingHost;
        break;
    }
    return geoServiceParameters;
}

DataObjectListModel *cterritories::getAddressTypes()
{
    sql_class *sql = &Singleton<sql_class>::Instance();
    int defaultlang = sql->getLanguageDefaultId();

    // list of address-types
    sql_items addressTypeItems = sql->selectSql("SELECT * FROM territory_addresstype WHERE active = 1 AND lang_id = '" + QVariant(defaultlang).toString() + "' ORDER BY addresstype_number;");
    if (addressTypeItems.size() == 0) {
        sql_item s;
        s.insert("addresstype_number", 1);
        s.insert("addresstype_name", tr("Do not call", "Territory address type"));
        s.insert("color", "#ff0000");
        s.insert("lang_id", defaultlang);
        sql->insertSql("territory_addresstype", &s, "id");
        addressTypeItems = sql->selectSql("SELECT * FROM territory_addresstype WHERE active = 1 AND lang_id = '" + QVariant(defaultlang).toString() + "' ORDER BY addresstype_number;");
    }
    DataObjectListModel *addressTypeListModel = new DataObjectListModel();
    for (unsigned int i = 0; i < addressTypeItems.size(); i++) {
        addressTypeListModel->addDataObject(DataObject(addressTypeItems[i].value("addresstype_number").toInt(),
                                                       addressTypeItems[i].value("addresstype_name").toString(),
                                                       addressTypeItems[i].value("color").toString()));
    }
    return addressTypeListModel;
}

DataObjectListModel *cterritories::getStreetTypes()
{
    sql_class *sql = &Singleton<sql_class>::Instance();

    // list of street-types
    sql_items streetTypeItems = sql->selectSql("SELECT * FROM territory_streettype WHERE active = 1 ORDER BY streettype_name;");

    DataObjectListModel *streetTypeListModel = new DataObjectListModel();
    for (unsigned int i = 0; i < streetTypeItems.size(); i++) {
        streetTypeListModel->addDataObject(DataObject(streetTypeItems[i].value("id").toInt(),
                                                      streetTypeItems[i].value("streettype_name").toString(),
                                                      streetTypeItems[i].value("color").toString()));
    }
    return streetTypeListModel;
}

/**
 * @brief getAllTerritories - Get all territories from database
 * @return - List of territories
 */
QVariantList cterritories::getAllTerritories(QVariantList territoryIds)
{
    sql_class *sql = &Singleton<sql_class>::Instance();
    QVariantList list;
    QString territoryIdFilter("");
    if (territoryIds.length() > 0) {
        territoryIdFilter += " AND id IN (";
        QString sep;
        for (int i = 0; i < territoryIds.length(); i++) {
            int id = territoryIds[i].value<int>();
            territoryIdFilter += sep + QVariant(id).toString();
            sep = ",";
        }
        territoryIdFilter += ")";
    }
    sql_items allTerritories = sql->selectSql("SELECT * FROM territory WHERE active" + territoryIdFilter + " ORDER BY territory_number");
    if (!allTerritories.empty()) {
        for (unsigned int i = 0; i < allTerritories.size(); i++) {
            sql_item item = allTerritories[i];
            Territory *t = new Territory(
                    item.value("id").toInt(),
                    item.value("territory_number").toInt(),
                    item.value("locality").toString(),
                    item.value("city_id").toInt(),
                    item.value("type_id").toInt(),
                    item.value("priority").toInt(),
                    item.value("remark").toString(),
                    item.value("wkt_geometry").toString(),
                    item.value("uuid").toString());
            list.append(QVariant::fromValue(t)); // qVariantFromValue(reinterpret_cast<Territory *>(t)));
        }
    }
    return list;
}

/**
 * @brief getTerritory - Get a territory object by name
 * @param locality - territory locality
 * @return - territory object or 0 if not exist
 */
Territory *cterritories::getTerritory(QString locality)
{
    sql_class *sql = &Singleton<sql_class>::Instance();
    QString sqlcommand;

    sqlcommand = QString("SELECT * FROM territory WHERE locality = '%1' AND active").arg(locality);

    sql_items ts = sql->selectSql(sqlcommand);
    if (!ts.empty()) {
        sql_item item = ts[0];
        Territory *t = new Territory(
                item.value("id").toInt(),
                item.value("territory_number").toInt(),
                item.value("locality").toString(),
                item.value("city_id").toInt(),
                item.value("type_id").toInt(),
                item.value("priority").toInt(),
                item.value("remark").toString(),
                item.value("wkt_geometry").toString(),
                item.value("uuid").toString());
        return t;
    } else {
        return nullptr;
    }
}

/**
 * @brief getTerritoryByNumber - Get a territory object by number
 * @param number - number of the territory
 * @return - territory object or 0 if not exist
 */
Territory *cterritories::getTerritoryByNumber(int number)
{
    sql_class *sql = &Singleton<sql_class>::Instance();

    sql_items ts = sql->selectSql("SELECT * FROM territory WHERE active AND territory_number = " + QString::number(number));
    if (!ts.empty()) {
        sql_item item = ts[0];
        Territory *t = new Territory(
                item.value("id").toInt(),
                item.value("territory_number").toInt(),
                item.value("locality").toString(),
                item.value("city_id").toInt(),
                item.value("type_id").toInt(),
                item.value("priority").toInt(),
                item.value("remark").toString(),
                item.value("wkt_geometry").toString(),
                item.value("uuid").toString());
        return t;
    } else {
        return nullptr;
    }
}

/**
 * @brief getTerritoryById - Get a territory object by id
 * @param id - Id in database
 * @return - territory object or 0 if not exist
 */
Territory *cterritories::getTerritoryById(int id)
{
    sql_class *sql = &Singleton<sql_class>::Instance();

    sql_items ts = sql->selectSql("territory", "id", QString::number(id), "");
    if (!ts.empty()) {
        sql_item item = ts[0];
        Territory *t = new Territory(
                item.value("id").toInt(),
                item.value("territory_number").toInt(),
                item.value("locality").toString(),
                item.value("city_id").toInt(),
                item.value("type_id").toInt(),
                item.value("priority").toInt(),
                item.value("remark").toString(),
                item.value("wkt_geometry").toString(),
                item.value("uuid").toString());
        return t;
    } else {
        return nullptr;
    }
}

/**
 * @brief removeTerritory - Remove territory from the database
 * @param id - Territory's id in the database
 * @return - success or failure
 */
bool cterritories::removeTerritory(int id)
{
    sql_class *sql = &Singleton<sql_class>::Instance();
    // deactive territory in table
    sql_item s;
    s.insert("active", 0);
    bool isTerritoryRemoved = sql->updateSql("territory", "id", QString::number(id), &s);
    if (isTerritoryRemoved) {
        emit boundaryGeometryChanged();
        sql->updateSql("territory_assignment", "territory_id", QString::number(id), &s);
        int addressCount = sql->selectScalar("select count(*) from territory_address where id = ? and active", QString::number(id), 0).toInt();
        if (addressCount > 0) {
            if (sql->updateSql("territory_address", "territory_id", QString::number(id), &s)) {
                emit addressGeometryChanged();
            }
        }
        int streetCount = sql->selectScalar("select count(*) from territory_street where id = ? and active", QString::number(id), 0).toInt();
        if (streetCount > 0) {
            if (sql->updateSql("territory_street", "territory_id", QString::number(id), &s)) {
                emit streetGeometryChanged();
            }
        }
    }
    return isTerritoryRemoved;
}

/**
 * @brief importKmlGeometry - Import territory boundaries from kml-file
 * @param fileUrl - file name
 * @param nameMatchField, descriptionMatchField - match kml and database fields
 *               0 = none
 *               1 = territory_number
 *               2 = locality
 * @return - number of imported addresses; -1 = failure
 */
int cterritories::importKmlGeometry(QUrl fileUrl, int nameMatchField, int descriptionMatchField, bool searchByDescription)
{
    if (!fileUrl.isValid())
        return -1;

    QString fileName = fileUrl.toLocalFile();

    QDomDocument doc("kmlfile");
    QFile file(fileName);
    if (!file.open(QIODevice::ReadOnly))
        return -1;
    if (!doc.setContent(&file)) {
        file.close();
        return -1;
    }
    file.close();

    QDomElement docElem = doc.documentElement();
    if (docElem.attribute("xmlns", "") != "http://www.opengis.net/kml/2.2")
        return -1;

    // Placemark-Tags
    QDomNodeList features = docElem.elementsByTagName("Placemark");

    sql_class *sql;
    sql = &Singleton<sql_class>::Instance();
    sql->startTransaction();

    int importCount = 0;

    for (int iFeature = 0; iFeature < features.count(); iFeature++) {
        QString featureName;
        QString featureDescription;

        QDomNode featureNode = features.at(iFeature);
        QDomNode featureChildNode = featureNode.firstChild();
        while (!featureChildNode.isNull()) {
            QDomElement element = featureChildNode.toElement();
            // name-Tag
            if (element.tagName().compare("name", Qt::CaseInsensitive) == 0)
                featureName = element.text().trimmed();
            // description-Tag
            if (element.tagName().compare("description", Qt::CaseInsensitive) == 0)
                featureDescription = element.text().trimmed();
            featureChildNode = featureChildNode.nextSibling();
        }

        // Polygon-Tag
        QDomNodeList polygons;
        QDomElement featureElement = featureNode.toElement();
        if (!featureElement.isNull())
            polygons = featureElement.elementsByTagName("Polygon"); // Polygon can be a child of MultiGeometry

        QString wktGeometry("");
        QString sep("");
        int iPolygon(0);
        for (int iPolygon = 0; iPolygon < polygons.count(); iPolygon++) {
            wktGeometry += sep + "(";
            QDomNode polygonNode = polygons.at(iPolygon);
            QDomNode polygonChildNode = polygonNode.firstChild();
            while (!polygonChildNode.isNull()) {
                QDomElement polygonChildElement = polygonChildNode.toElement();
                // outerBoundary coordinates
                if (polygonChildElement.tagName().compare("outerBoundaryIs", Qt::CaseInsensitive) == 0) {
                    QDomNode boundaryNode = polygonChildElement.firstChild();
                    QDomNodeList outerBoundaries;
                    while (!boundaryNode.isNull()) {
                        QDomElement boundaryElement = boundaryNode.toElement();
                        if (boundaryElement.tagName().compare("LinearRing", Qt::CaseInsensitive) == 0) {
                            outerBoundaries = boundaryElement.elementsByTagName("coordinates");
                        }
                        boundaryNode = boundaryNode.nextSibling();
                    }

                    if (outerBoundaries.count() > 0) {
                        QDomNode coordinatesNode = outerBoundaries.at(0);
                        QDomElement coordinatesElement = coordinatesNode.toElement();
                        QStringList newGeometry;
                        QStringList coordinates = coordinatesElement.text().split(" ");
                        for (int i = 0; i < coordinates.count(); i++) {
                            QStringList values = coordinates[i].split(",");
                            if (values.size() >= 2) {
                                newGeometry << values[0].trimmed() + " " + values[1].trimmed();
                            }
                        }
                        wktGeometry += "(" + newGeometry.join(", ") + ")";
                    }
                }
                sep = ",";

                // innerBoundary coordinates
                if (polygonChildElement.tagName().compare("innerBoundaryIs", Qt::CaseInsensitive) == 0) {
                    QDomNode boundaryNode = polygonChildElement.firstChild();
                    QDomNodeList innerBoundaries;
                    while (!boundaryNode.isNull()) {
                        QDomElement boundaryElement = boundaryNode.toElement();
                        if (boundaryElement.tagName().compare("LinearRing", Qt::CaseInsensitive) == 0) {
                            innerBoundaries = boundaryElement.elementsByTagName("coordinates");
                        }
                        boundaryNode = boundaryNode.nextSibling();
                    }

                    for (int i = 0; i < innerBoundaries.count(); i++) {
                        QDomNode boundaryNode = innerBoundaries.at(i);
                        QDomElement boundaryElement = boundaryNode.toElement();
                        QStringList newGeometry;
                        QStringList coordinates = boundaryElement.text().split(" ");
                        for (int i = 0; i < coordinates.count(); i++) {
                            QStringList values = coordinates[i].split(",");
                            if (values.size() >= 2) {
                                newGeometry << values[0].trimmed() + " " + values[1].trimmed();
                            }
                        }
                        wktGeometry += sep + "(" + newGeometry.join(", ") + ")";
                        sep = ",";
                    }
                }

                polygonChildNode = polygonChildNode.nextSibling();
            }
            wktGeometry += ")";
        }

        if (iPolygon > 1)
            wktGeometry = "MULTIPOLYGON (" + wktGeometry + ")";
        else
            wktGeometry = "POLYGON " + wktGeometry;

        Territory *t = nullptr;
        switch (nameMatchField) {
        case 1:
            t = getTerritoryByNumber(QVariant(featureName).toInt());
            break;
        case 2:
            t = getTerritory(featureName);
            break;
        }

        if (searchByDescription && !t) {
            switch (descriptionMatchField) {
            case 1:
                t = getTerritoryByNumber(QVariant(featureDescription).toInt());
                break;
            case 2:
                t = getTerritory(featureDescription);
                break;
            }
        }
        if (!t) {
            t = new Territory();
            t->setTerritoryNumber(-1);
        }
        if (t) {
            t->setWktGeometry(wktGeometry);
            switch (nameMatchField) {
            case 1:
                t->setTerritoryNumber(QVariant(featureName).toInt());
                break;
            case 2:
                t->setLocality(featureName);
                break;
            case 3:
                t->setRemark(featureName);
                break;
            }
            switch (descriptionMatchField) {
            case 1:
                t->setTerritoryNumber(QVariant(featureDescription).toInt());
                break;
            case 2:
                t->setLocality(featureDescription);
                break;
            case 3:
                t->setRemark(featureDescription);
                break;
            }
            if (t->save())
                importCount += 1;
        }
    }

    sql->commitTransaction();
    if (importCount > 0)
        emit boundaryGeometryChanged();
    return importCount;
}

/**
 * @brief getTerritoryBoundaries - Loads the territory boundaries from the database and emits a boundariesLoaded signal.
 * @param territoryId
 *               0 = default; load all boundaries
 *               n = load boundaries of the given territory
 */
void cterritories::getTerritoryBoundaries(int territoryId)
{
    sql_class *sql = &Singleton<sql_class>::Instance();
    QString sqlQuery = "SELECT id, wkt_geometry FROM territories WHERE wkt_geometry IS NOT NULL AND wkt_geometry <> ''";
    sql_items territories = sql->selectSql(sqlQuery);

    QVariantMap boundaryMap;
    QVariantMap allPointsMap;
    if (!territories.empty()) {
        for (unsigned int i = 0; i < territories.size(); i++) {
            const sql_item territory = territories[i];
            int territory_id = territory.value("id").toInt();
            QString input = territory.value("wkt_geometry").toString();
            QRegularExpressionMatchIterator iMultiPolygon = multiPolygonRegExp->globalMatch(input);
            while (iMultiPolygon.hasNext()) {
                QRegularExpressionMatch multiPolygonMatch = iMultiPolygon.next();
                QString multiPolygon = multiPolygonMatch.captured(0);
                QRegularExpressionMatchIterator iPolygon = polygonRegExp->globalMatch(multiPolygon);

                Boundary boundary;
                boundary.setTerritoryId(territory_id);
                QGeoPolygon geoPolygon;
                int innerPolygonIndex = 0;
                while (iPolygon.hasNext()) {
                    QRegularExpressionMatch polygonMatch = iPolygon.next();
                    QString polygon = polygonMatch.captured(0);
                    QRegularExpressionMatchIterator iMatch = coordinateRegExp->globalMatch(polygon);
                    QList<QGeoCoordinate> coordinates;
                    while (iMatch.hasNext()) {
                        QRegularExpressionMatch match = iMatch.next();
                        double longitude = match.captured(1).toDouble();
                        double latitude = match.captured(2).toDouble();
                        QGeoCoordinate coordinate(latitude, longitude);
                        coordinates.append(coordinate);
                    }
                    if (innerPolygonIndex == 0) {
                        // fetched coordinates belong to current polygon's perimeter
                        geoPolygon.setPerimeter(coordinates);
                    } else {
                        // fetched coordinates belong to a hole in the current polygon
                        geoPolygon.addHole(coordinates);
                    }
                    innerPolygonIndex += 1;
                }
                boundary.setGeoPolygon(geoPolygon);

                // workaround for territory within boundary of another territory:
                // boundaries are sorted in reverse order by combination of area size and polygon-index in geometry definition
                // in order to ensure that smaller territories are on top of larger territories
                double width = geoPolygon.boundingGeoRectangle().width();
                double height = geoPolygon.boundingGeoRectangle().height();
                QString sortOrder = QString::number(9999999 - width * height, 'f', 9) + "_" + QString::number(999 - innerPolygonIndex);

                boundaryMap[sortOrder + "_" + QString::number(territory_id)] = QVariant::fromValue(boundary);
            }
        }
    }
    emit boundariesLoaded(boundaryMap, territoryId);
}

#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
// GDAL missing
#elif defined(Q_OS_LINUX) || defined(Q_OS_MAC) || defined(Q_OS_WIN)
/**
 * @brief setTerritoryBoundary - Set and save the boundary of a territory.
 * @param territoryId - territory id
 * @param path - geometry path
 * @param dissolve - if true old and new boundaries are dissolved, otherwise old boundary is replaced
 * @param overlappedTerritoryIds - ids of the territories which intersect the given path
 */
bool cterritories::setTerritoryBoundary(int territoryId, QList<QGeoCoordinate> path, bool dissolve, QVariantList overlappedTerritoryIds, bool annexOverlappingAreas)
{
    // convert path
    OGRSpatialReference osr;
    OGRPolygon polygon;
    OGRLinearRing poRing;
    for (auto c = path.begin(), end = path.end(); c != end; ++c)
        poRing.addPoint((*c).longitude(), (*c).latitude());
    poRing.closeRings();
    polygon.addRing(&poRing);
    OGRGeometry *newGeometry = polygon.getLinearGeometry();

    int changes = setTerritoryBoundary(territoryId, newGeometry, dissolve, overlappedTerritoryIds, annexOverlappingAreas);
    if (changes & 1)
        emit boundaryGeometryChanged();
    if (changes & 2)
        emit streetGeometryChanged();
    if (changes & 4)
        emit addressGeometryChanged();
    return changes & 1;
}

bool cterritories::setTerritoryBoundary(int territoryId, QString wktGeometry, bool dissolve, QVariantList overlappedTerritoryIds, bool annexOverlappingAreas)
{
    // convert path
    OGRSpatialReference osr;
    OGRGeometry *newGeometry = nullptr;
    QByteArray bytes = wktGeometry.toUtf8();
    const char *data = bytes.constData();
    OGRErr err = OGRGeometryFactory::createFromWkt(data, &osr, &newGeometry);
    if (err != OGRERR_NONE) {
        // process error, like emit signal
        return false;
    } else {
        int changes = setTerritoryBoundary(territoryId, newGeometry, dissolve, overlappedTerritoryIds, annexOverlappingAreas);
        if (changes & 1)
            emit boundaryGeometryChanged();
        if (changes & 2)
            emit streetGeometryChanged();
        if (changes & 4)
            emit addressGeometryChanged();
        return changes & 1;
    }
}

int cterritories::setTerritoryBoundary(int territoryId, OGRGeometry *newGeometry, bool dissolve, QVariantList overlappedTerritoryIds, bool annexOverlappingAreas)
{
    bool territoryChanged = false;
    bool streetChanged = false;
    bool addressChanged = false;
    Territory *t = nullptr;
    t = getTerritoryById(territoryId);
    if (t) {
        OGRSpatialReference osr;
        OGRPolygon polygon;

        if (dissolve) {
            OGRGeometry *oldTerritoryGeometry = t->getOGRGeometry();
            OGRGeometry *dissolvedTerritoryGeometry = nullptr;

            if (oldTerritoryGeometry == nullptr || oldTerritoryGeometry->IsEmpty())
                dissolvedTerritoryGeometry = newGeometry->clone();
            else {
                if (!oldTerritoryGeometry->IsValid()) {
                    // TODO: try to repair the geometry (self-intersecting polygon, ...)
                    // wait until GDAL 3.0 and GEOS 3.8 is available on all systems
                    //                    oldTerritoryGeometry = oldTerritoryGeometry->MakeValid();
                    //                    if (oldTerritoryGeometry != nullptr)
                    //                        oldTerritoryGeometry = oldTerritoryGeometry->Buffer(0.0);
                    return 0;
                }
                if (oldTerritoryGeometry != nullptr && oldTerritoryGeometry->IsValid())
                    dissolvedTerritoryGeometry = oldTerritoryGeometry->Union(newGeometry);
                else
                    return 0;
            }
            if (dissolvedTerritoryGeometry != nullptr)
                newGeometry = dissolvedTerritoryGeometry;
            else
                return 0;
        }

        if (overlappedTerritoryIds.length() > 0) {
            // remove overlapping areas
            OGRGeometry *tmpTerritoryGeometry = newGeometry->clone();
            QVariantList overlappedTerritories = getAllTerritories(overlappedTerritoryIds);
            for (int i = 0; i < overlappedTerritories.length(); i++) {
                Territory *overlappedTerritory = overlappedTerritories[i].value<Territory *>();
                if (overlappedTerritory->territoryId() != territoryId) {
                    OGRGeometry *overlappedTerritoryGeometry = overlappedTerritory->getOGRGeometry();
                    if (overlappedTerritoryGeometry != nullptr && !overlappedTerritoryGeometry->IsEmpty())
                        tmpTerritoryGeometry = tmpTerritoryGeometry->Difference(overlappedTerritoryGeometry);
                }
            }

            if (annexOverlappingAreas) {
                // save new boundary before annexing overlapping areas
                qDebug() << "Save new boundary of territory" << t->territoryNumber() << "before annexing overlapping areas.";
                if (!t->setOGRGeometry(tmpTerritoryGeometry))
                    return 0;
                if (t->isDirty())
                    territoryChanged |= t->save();

                // annex overlaping areas
                sql_class *sql = &Singleton<sql_class>::Instance();
                sql->startTransaction();
                bool successfullyAnnexed = true;
                for (int i = 0; i < overlappedTerritories.length(); i++) {
                    Territory *overlappedTerritory = overlappedTerritories[i].value<Territory *>();
                    if (overlappedTerritory->territoryId() != territoryId) {
                        int changes = annexArea(territoryId, overlappedTerritory->territoryId(), newGeometry);
                        successfullyAnnexed = changes & 1;
                        if (successfullyAnnexed) {
                            streetChanged |= changes & 2;
                            addressChanged |= changes & 4;
                        }
                    }
                    if (!successfullyAnnexed)
                        break;
                }
                if (!successfullyAnnexed)
                    sql->rollbackTransaction();
                else
                    sql->commitTransaction();

                newGeometry = t->getOGRGeometry();
            } else {
                newGeometry = tmpTerritoryGeometry;
            }
        }

        // save new boundary
        if (t->setOGRGeometry(newGeometry)) {
            if (t->isDirty())
                territoryChanged |= t->save();
        }

        // add streets and addresses of territories without boundary if they are within the current territory's area
        sql_class *sql = &Singleton<sql_class>::Instance();
        QString sqlQuery = "SELECT * FROM territory WHERE wkt_geometry IS NULL AND active";
        sql_items territoryWithoutBoundaryRows = sql->selectSql(sqlQuery);
        if (!territoryWithoutBoundaryRows.empty()) {
            for (unsigned int i = 0; i < territoryWithoutBoundaryRows.size(); i++) {
                sql_item territoryWithoutBoundaryRow = territoryWithoutBoundaryRows[i];
                streetChanged |= moveStreetsToTerritory(territoryWithoutBoundaryRow.value("id").toInt(), territoryId);
                addressChanged |= moveAddressesToTerritory(territoryWithoutBoundaryRow.value("id").toInt(), territoryId);
            }
        }
        return (addressChanged << 2) | (streetChanged << 1) | territoryChanged;
    }
    return 0;
}

/**
 * @brief splitTerritory - Split territory using a polygon which defines the area that is cut off and created as a new territory
 * @param territoryId - territory id
 * @param cutAreaPath - polygon path of the cut area
 * @return - id of the newly created territory
 */
int cterritories::splitTerritory(int territoryId, QList<QGeoCoordinate> cutAreaPath)
{
    int newTerritoryId = 0;
    Territory *t = nullptr;
    t = getTerritoryById(territoryId);
    if (!t)
        return 0;

    OGRSpatialReference osr;

    // convert polygon path
    OGRLinearRing poRing;

    for (auto c = cutAreaPath.begin(), end = cutAreaPath.end(); c != end; ++c)
        poRing.addPoint((*c).longitude(), (*c).latitude());

    poRing.closeRings();
    OGRPolygon cutAreaPolygon;
    OGRGeometry *cutAreaGeometry = nullptr;
    cutAreaPolygon.addRing(&poRing);
    cutAreaGeometry = cutAreaPolygon.getLinearGeometry();

    OGRGeometry *territoryGeometry = t->getOGRGeometry();
    if (territoryGeometry == nullptr)
        return 0;

    // create new territory and annex intersection with cut area
    OGRGeometry *insideCutAreaGeometry = nullptr;
    insideCutAreaGeometry = territoryGeometry->Intersection(cutAreaGeometry);
    sql_class *sql = &Singleton<sql_class>::Instance();
    sql->startTransaction();
    int changes = annexArea(newTerritoryId, territoryId, insideCutAreaGeometry);
    if (changes & 1) {
        sql->commitTransaction();
        emit boundaryGeometryChanged();
        if (changes & 2)
            emit streetGeometryChanged();
        if (changes & 4)
            emit addressGeometryChanged();
        return newTerritoryId;
    } else {
        sql->rollbackTransaction();
        return 0;
    }
}
#endif

QString cterritories::getCongregationAddress() const
{
    ccongregation c;
    ccongregation::congregation myCongregation;
    myCongregation = c.getMyCongregation();
    return myCongregation.address;
}

void cterritories::geocodeAddress(const QVariant &address, const QGeoShape &bounds)
{
    Q_UNUSED(bounds);
    QSettings settings;
    // Deactivate default Qt geocoding as it doesn't return the house number
    // in the corresponding streetNumber field (e.g. in Germany)
    // int service = settings.value("geo_service_provider/default", 0).toInt();
    // if (service == 0) {
    //     // default Qt geocoding
    //     if (address.canConvert<QGeoAddress>()) {
    //         QGeoAddress geoAddress;
    //         geoAddress = address.value<QGeoAddress>();
    //         // geocodingManager->geocode(geoAddress, bounds);
    //         // try with plain text
    //         QString addressText = geoAddress.text();
    //         addressText = addressText.replace(QRegularExpression("<[^>]*>"), " ").toHtmlEscaped();
    //         geocodingManager->geocode(addressText, -1, 0, bounds);
    //     } else {
    //         geocodingManager->geocode(address.toString(), -1, 0, bounds);
    //     }
    // } else {
    // alternative geocoding
    QString addressText;
    if (address.canConvert<QGeoAddress>()) {
        QGeoAddress geoAddress;
        geoAddress = address.value<QGeoAddress>();
        addressText = geoAddress.text();
    } else
        addressText = address.toString();
    addressText = addressText.replace(QRegularExpression("<[^>]*>"), " ").toHtmlEscaped();
    QUrl geoCodeUrl = getGeocodeUrl(addressText);
    if (geoCodeUrl.isValid()) {
        QNetworkRequest request;
        request.setUrl(geoCodeUrl);
        addressRequest->get(request);
    }
    // }
}

void cterritories::geocodeReplyFinished(QGeoCodeReply *reply)
{
    if (reply->error() != QGeoCodeReply::NoError) {
        reply->deleteLater();
        emit geocodingError(reply->errorString());
        return;
    }

    QVariantList geocodeVariantList;
    for (int i = 0; i < reply->locations().length(); i++) {
        QGeoLocation geoLocation = reply->locations()[i];
        QVariantMap extAttributes;
        if (!geoLocation.address().isTextGenerated()) {
            // save formatted address received from the geocoding service
            // and reset text to generate simple postal address
            QGeoAddress geoAddress = geoLocation.address();
            QString formattedAddress = geoAddress.text();
            geoAddress.setText("");
            geoLocation.setAddress(geoAddress);
            extAttributes.insert("formattedAddress", formattedAddress);
        }
        // create WKT
        QString wktGeometry = "POINT("
                + QVariant(geoLocation.coordinate().longitude()).toString() + " "
                + QVariant(geoLocation.coordinate().latitude()).toString() + ")";
        extAttributes.insert("wktGeometry", wktGeometry);
        geoLocation.setExtendedAttributes(extAttributes);

        geocodeVariantList.append(QVariant::fromValue(geoLocation));
    }
    emit geocodingFinished(geocodeVariantList);
}

void cterritories::addressRequestFinished(QNetworkReply *reply)
{
    QJsonDocument doc = QJsonDocument::fromJson(reply->readAll());
    if (reply->error() != QNetworkReply::NoError) {
        reply->close();
        emit geocodingError(reply->errorString());
        return;
    }

    QList<QGeoLocation *> geocodeResultList = getGeocodeResults(doc);
    QVariantList geocodeVariantList;
    for (int i = 0; i < geocodeResultList.length(); i++) {
        geocodeVariantList.append(QVariant::fromValue(*geocodeResultList[i]));
    }
    emit geocodingFinished(geocodeVariantList);
}

#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
// GDAL missing
#elif defined(Q_OS_LINUX) || defined(Q_OS_MAC) || defined(Q_OS_WIN)
void cterritories::streetRequestFinished(QNetworkReply *reply)
{
    int territoryId = reply->property("territoryId").toInt();
    QJsonDocument doc = QJsonDocument::fromJson(reply->readAll());
    if (reply->error() != QNetworkReply::NoError) {
        reply->close();
        emit streetRequestFailed(reply->errorString());
        return;
    }

    QList<StreetResult *> streetResultList = getStreetResults(doc, territoryId);
    QVariantList streetVariantList;
    for (int i = 0; i < streetResultList.length(); i++) {
        // streetVariantList.append(qVariantFromValue(reinterpret_cast<StreetResult *>(streetResultList[i])));
        streetVariantList.append(QVariant::fromValue(streetResultList[i]));
    }

    emit streetListReceived(streetVariantList);
}
#endif

QUrl cterritories::getGeocodeUrl(QString address)
{
    QSettings settings;
    int service = settings.value("geo_service_provider/default", 0).toInt();
    switch (service) {
    case 1: {
        // Google geocoding service
        QString googleAPIkey = settings.value("geo_service_provider/google_api_key", "").toString();
        googleAPIkey = googleAPIkey.isEmpty() ? "" : "&key=" + googleAPIkey;
        return QUrl("https://maps.googleapis.com/maps/api/geocode/json?address=" + address + "&sensor=false" + googleAPIkey);
    }
    case 2: {
        // Here geocoding service
        QString hereAPIKey = settings.value("geo_service_provider/here_api_key", "").toString();
        hereAPIKey = hereAPIKey.isEmpty() ? "" : "&apiKey=" + hereAPIKey;
        return QUrl("https://geocoder.api.here.com/6.2/geocode.json?&searchtext=" + address + hereAPIKey);
    }
    default:
        // OSM geocoding service
        return QUrl("https://nominatim.openstreetmap.org/?format=json&addressdetails=1&q=" + address);
    }
}

QList<QGeoLocation *> cterritories::getGeocodeResults(QJsonDocument &doc)
{
    QList<QGeoLocation *> geocodeResultList;

    QSettings settings;
    int service = settings.value("geo_service_provider/default", 0).toInt();
    switch (service) {
    case 1: {
        // Google geocoding service
        if (doc.isObject()) {
            QJsonObject obj = doc.object();

            QString status = obj.value(QStringLiteral("status")).toString();
            if (status == "OK") {
                QJsonArray results = obj.value(QStringLiteral("results")).toArray();

                for (int iResult = 0; iResult < results.size(); iResult++) {
                    QGeoLocation *geoLocation = new QGeoLocation();
                    QGeoAddress geoAddress;
                    QJsonObject result = results.at(iResult).toObject();
                    QJsonArray addressComponents = result.value(QStringLiteral("address_components")).toArray();

                    for (int iAddressComp = 0; iAddressComp < addressComponents.size(); iAddressComp++) {
                        QJsonObject addressComponent = addressComponents.at(iAddressComp).toObject();
                        QJsonArray addressTypes = addressComponent.value(QStringLiteral("types")).toArray();
                        QJsonValue addressType("country");
                        if (addressTypes.contains(addressType))
                            geoAddress.setCountry(addressComponent.value(QStringLiteral("short_name")).toString());

                        addressType = "administrative_area_level_1";
                        if (addressTypes.contains(addressType))
                            geoAddress.setState(addressComponent.value(QStringLiteral("short_name")).toString());

                        addressType = "administrative_area_level_2";
                        if (addressTypes.contains(addressType))
                            geoAddress.setCounty(addressComponent.value(QStringLiteral("long_name")).toString());

                        addressType = "locality";
                        if (addressTypes.contains(addressType))
                            geoAddress.setCity(addressComponent.value(QStringLiteral("long_name")).toString());

                        addressType = "sublocality";
                        if (addressTypes.contains(addressType))
                            geoAddress.setDistrict(addressComponent.value(QStringLiteral("long_name")).toString());

                        addressType = "route";
                        if (addressTypes.contains(addressType))
                            geoAddress.setStreet(addressComponent.value(QStringLiteral("long_name")).toString());

                        addressType = "street_number";
                        if (addressTypes.contains(addressType))
                            geoAddress.setStreetNumber(addressComponent.value(QStringLiteral("long_name")).toString());

                        addressType = "postal_code";
                        if (addressTypes.contains(addressType))
                            geoAddress.setPostalCode(addressComponent.value(QStringLiteral("long_name")).toString());
                    }
                    geoAddress.setText("");
                    geoLocation->setAddress(geoAddress);

                    QJsonObject jsonGeometry = result.value(QStringLiteral("geometry")).toObject();
                    QJsonObject jsonLocation = jsonGeometry.value(QStringLiteral("location")).toObject();
                    QGeoCoordinate geoCoordinate;
                    double lat = jsonLocation.value(QStringLiteral("lat")).toDouble();
                    geoCoordinate.setLatitude(lat);
                    double lon = jsonLocation.value(QStringLiteral("lng")).toDouble();
                    geoCoordinate.setLongitude(lon);
                    geoLocation->setCoordinate(geoCoordinate);

                    QVariantMap extAttributes;
                    QString wktGeometry = "POINT(" + QVariant(lon).toString() + " " + QVariant(lat).toString() + ")";
                    extAttributes.insert("wktGeometry", wktGeometry);
                    QString formattedAddress = result.value(QStringLiteral("formatted_address")).toString();
                    extAttributes.insert("formattedAddress", formattedAddress);
                    geoLocation->setExtendedAttributes(extAttributes);

                    geocodeResultList.append(geoLocation);
                }
            } else {
                QString errorMessage = obj.value(QStringLiteral("error_message")).toString();
                emit geocodingError(errorMessage);
            }
        }
        break;
    }
    case 2: {
        // Here geocoding service
        if (doc.isObject()) {
            QJsonObject obj = doc.object();
            QJsonObject response = obj.value(QStringLiteral("Response")).toObject();
            QJsonArray views = response.value(QStringLiteral("View")).toArray();
            for (int iView = 0; iView < views.size(); iView++) {
                QJsonObject view = views.at(iView).toObject();
                QJsonArray results = view.value(QStringLiteral("Result")).toArray();
                for (int iResult = 0; iResult < results.size(); iResult++) {
                    QGeoLocation *geoLocation = new QGeoLocation();
                    QGeoAddress geoAddress;

                    QJsonObject result = results.at(iResult).toObject();
                    QJsonObject jsonLocation = result.value(QStringLiteral("Location")).toObject();

                    QJsonObject jsonAddress = jsonLocation.value(QStringLiteral("Address")).toObject();
                    geoAddress.setCountry(jsonAddress.value(QStringLiteral("Country")).toString());
                    geoAddress.setState(jsonAddress.value(QStringLiteral("State")).toString());
                    geoAddress.setCounty(jsonAddress.value(QStringLiteral("County")).toString());
                    geoAddress.setCity(jsonAddress.value(QStringLiteral("City")).toString());
                    geoAddress.setDistrict(jsonAddress.value(QStringLiteral("District")).toString());
                    geoAddress.setStreet(jsonAddress.value(QStringLiteral("Street")).toString());
                    geoAddress.setStreetNumber(jsonAddress.value(QStringLiteral("HouseNumber")).toString());
                    geoAddress.setPostalCode(jsonAddress.value(QStringLiteral("PostalCode")).toString());
                    geoAddress.setText("");
                    geoLocation->setAddress(geoAddress);

                    QGeoCoordinate geoCoordinate;
                    QJsonObject displayPosition = jsonLocation.value(QStringLiteral("DisplayPosition")).toObject();
                    double lat = displayPosition.value(QStringLiteral("Latitude")).toDouble();
                    geoCoordinate.setLatitude(lat);
                    double lon = displayPosition.value(QStringLiteral("Longitude")).toDouble();
                    geoCoordinate.setLongitude(lon);
                    geoLocation->setCoordinate(geoCoordinate);

                    QVariantMap extAttributes;
                    QString wktGeometry = "POINT(" + QVariant(lon).toString() + " " + QVariant(lat).toString() + ")";
                    extAttributes.insert("wktGeometry", wktGeometry);
                    QString formattedAddress = jsonAddress.value(QStringLiteral("Label")).toString();
                    extAttributes.insert("formattedAddress", formattedAddress);
                    geoLocation->setExtendedAttributes(extAttributes);

                    geocodeResultList.append(geoLocation);
                }
            }
        }
        break;
    }
    default:
        // OSM geocoding service
        if (doc.isArray()) {
            QJsonArray results = doc.array();
            for (int iResult = 0; iResult < results.size(); iResult++) {
                QGeoLocation *geoLocation = new QGeoLocation();
                QJsonObject result = results.at(iResult).toObject();

                QJsonObject jsonAddress = result.value(QStringLiteral("address")).toObject();
                QGeoAddress geoAddress;
                geoAddress.setCountry(jsonAddress.value(QStringLiteral("country_code")).toString());
                geoAddress.setState(jsonAddress.value(QStringLiteral("state")).toString());
                geoAddress.setCounty(jsonAddress.value(QStringLiteral("county")).toString());
                geoAddress.setCity(jsonAddress.value(QStringLiteral("city")).toString());
                geoAddress.setDistrict(jsonAddress.value(QStringLiteral("suburb")).toString());
                geoAddress.setStreet(jsonAddress.value(QStringLiteral("road")).toString());
                geoAddress.setStreetNumber(jsonAddress.value(QStringLiteral("house_number")).toString());
                geoAddress.setPostalCode(jsonAddress.value(QStringLiteral("postcode")).toString());
                geoAddress.setText("");
                geoLocation->setAddress(geoAddress);

                double lat = result.value(QStringLiteral("lat")).toString().toDouble();
                double lon = result.value(QStringLiteral("lon")).toString().toDouble();
                QGeoCoordinate geoCoordinate;
                geoCoordinate.setLatitude(lat);
                geoCoordinate.setLongitude(lon);
                geoLocation->setCoordinate(geoCoordinate);

                QVariantMap extAttributes;
                QString wktGeometry = "POINT(" + QVariant(lon).toString() + " " + QVariant(lat).toString() + ")";
                extAttributes.insert("wktGeometry", wktGeometry);
                QString formattedAddress = result.value(QStringLiteral("display_name")).toString();
                extAttributes.insert("formattedAddress", formattedAddress);
                geoLocation->setExtendedAttributes(extAttributes);

                geocodeResultList.append(geoLocation);
            }
        }
        break;
    }
    return geocodeResultList;
}

#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
// GDAL missing
#elif defined(Q_OS_LINUX) || defined(Q_OS_MAC) || defined(Q_OS_WIN)
QList<StreetResult *> cterritories::getStreetResults(QJsonDocument &doc, int territoryId)
{
    QList<StreetResult *> streetResultList;
    Territory *t = getTerritoryById(territoryId);
    if (!t)
        return streetResultList;
    // Parse WKT boundary of the territory for intersection
    QString wktGeometry = t->wktGeometry();
    OGRSpatialReference osr;
    OGRGeometry *territoryGeometry = nullptr;
    QByteArray bytes = wktGeometry.toUtf8();
    const char *data = bytes.constData();
    OGRErr err = OGRGeometryFactory::createFromWkt(data, &osr, &territoryGeometry);
    if (err != OGRERR_NONE) {
        // process error, like emit signal
        return streetResultList;
    }

    // OSM overpass service
    if (doc.isObject()) {
        QJsonObject obj = doc.object();
        QJsonArray elements = obj.value(QStringLiteral("elements")).toArray();

        // create a list of already added streets for checking if new streets are already added
        QList<QString> territoryStreetList;
        sql_class *sql = &Singleton<sql_class>::Instance();
        QString sqlQuery = "SELECT street_name FROM territorystreets WHERE territory_id = "
                + QVariant(t->territoryId()).toString()
                + " GROUP BY street_name";
        sql_items streets = sql->selectSql(sqlQuery);
        for (sql_item &street : streets) {
            territoryStreetList.append(street.value("street_name").toString().toLower());
        }

        for (int iElement = 0; iElement < elements.size(); iElement++) {
            QJsonObject element = elements.at(iElement).toObject();
            QJsonObject tags = element.value(QStringLiteral("tags")).toObject();
            QString streetName = tags.value(QStringLiteral("name")).toString();

            // create linestring from current geometry
            QJsonArray points = element.value(QStringLiteral("geometry")).toArray();
            OGRLineString *lineString = new OGRLineString();
            for (int iPoint = 0; iPoint < points.size(); iPoint++) {
                QJsonObject point = points.at(iPoint).toObject();
                lineString->addPoint(point.value(QStringLiteral("lon")).toDouble(),
                                     point.value(QStringLiteral("lat")).toDouble());
            }
            StreetResult *streetResult = nullptr;
            // search for previously added street geometry part
            for (int iStreet = 0; iStreet < streetResultList.size(); iStreet++) {
                if (streetResultList[iStreet]->streetName() == streetName) {
                    streetResult = streetResultList[iStreet];
                    break;
                }
            }
            char *wkt_new = nullptr;
            OGRGeometry *streetGeometry = nullptr;
            streetGeometry = territoryGeometry->Intersection(lineString);
            if (streetGeometry && !streetGeometry->IsEmpty()) {
                if (!streetResult) {
                    // set street geometry
                    streetResult = new StreetResult(this);
                    streetResult->setStreetName(streetName);
                    streetGeometry->exportToWkt(&wkt_new);
                    streetResult->setWktGeometry(QString(wkt_new));
                    streetResult->setIsAlreadyAdded(territoryStreetList.contains(streetName.toLower()));
                    streetResultList.append(streetResult);
                } else {
                    // append street geometry to previously added
                    OGRMultiLineString *multiLineString = new OGRMultiLineString();
                    // import existing geometry
                    QByteArray bytes = streetResult->wktGeometry().toUtf8();
                    const char *data = bytes.constData();
                    OGRErr importError = multiLineString->importFromWkt(&data);
                    if (importError != OGRERR_NONE) {
                        OGRGeometry *existingGeometry = nullptr;
                        // Parse WKT
                        importError = OGRGeometryFactory::createFromWkt(data, &osr, &existingGeometry);
                        if (importError == OGRERR_NONE) {
                            importError = multiLineString->addGeometry(existingGeometry);
                        }
                        if (importError == OGRERR_NONE) {
                            // failed to create geometry from what was added previously
                        }
                    }
                    // add new street section
                    multiLineString->addGeometry(streetGeometry);
                    multiLineString->exportToWkt(&wkt_new);
                    streetResult->setWktGeometry(QString(wkt_new));
                }
            }
        }
    }
    std::sort(streetResultList.begin(), streetResultList.end(),
              [](const StreetResult *a, const StreetResult *b) {
                  return QString::localeAwareCompare(a->streetName(), b->streetName()) < 0;
              });
    return streetResultList;
}

/**
 * @brief annexArea - Annex area of another territory
 * @param territoryId - territory into which area is annexed; if id is 0, a new territory is created
 * @param fromTerritoryId - disconnect area from given territory
 * @param areaGeometry - polygon path of the cut area
 * @return - success
 */
int cterritories::annexArea(int &intoTerritoryId, int fromTerritoryId, OGRGeometry *areaGeometry)
{
    bool territoryChanged = false;
    bool streetChanged = false;
    bool addressChanged = false;
    Territory *fromTerritory = nullptr;
    fromTerritory = getTerritoryById(fromTerritoryId);
    if (!fromTerritory || areaGeometry == nullptr || !areaGeometry->IsValid())
        return 0;

    OGRSpatialReference osr;
    OGRGeometry *fromTerritoryGeometry = fromTerritory->getOGRGeometry();
    OGRGeometry *intoTerritoryGeometry = nullptr;

    sql_class *sql = &Singleton<sql_class>::Instance();

    // cut and update territory's boundary
    OGRGeometry *outsideAreaGeometry = nullptr;
    outsideAreaGeometry = fromTerritoryGeometry->Difference(areaGeometry);
    qDebug() << "Cut and update boundary of territory" << fromTerritory->territoryNumber();
    if (fromTerritory->setOGRGeometry(outsideAreaGeometry))
        territoryChanged |= fromTerritory->save();

    // get intersecting area which will be annexed
    OGRGeometry *insideAreaGeometry = nullptr;
    insideAreaGeometry = fromTerritoryGeometry->Intersection(areaGeometry);
    if (insideAreaGeometry->IsEmpty()) {
        qDebug() << "Intersecting area with territory" << fromTerritory->territoryNumber() << "is empty!";
        return 0;
    }
    if (insideAreaGeometry->IsValid()) {
        switch (insideAreaGeometry->getGeometryType()) {
        case wkbPolygon:
            break;
        case wkbMultiPolygon:
            insideAreaGeometry = insideAreaGeometry->UnionCascaded();
            break;
        case wkbGeometryCollection: {
            qDebug() << "Clean up geometry collection of intersection with territory" << fromTerritory->territoryNumber();
            OGRGeometryCollection *poColl = insideAreaGeometry->toGeometryCollection();
            OGRGeometry *validGeometry = nullptr;
            for (auto &&poSubGeom : *poColl) {
                switch (poSubGeom->getGeometryType()) {
                case wkbPolygon:
                    if (validGeometry == nullptr)
                        validGeometry = poSubGeom;
                    else
                        validGeometry = validGeometry->Union(poSubGeom);
                    break;
                case wkbMultiPolygon:
                    if (validGeometry == nullptr)
                        validGeometry = poSubGeom;
                    else
                        validGeometry = validGeometry->Union(poSubGeom);
                    break;
                case wkbLineString:
                    break;
                default:
                    qDebug() << "Subgeometry type is" << poSubGeom->getGeometryName();
                    break;
                }
            }

            if (validGeometry->getGeometryType() == wkbMultiPolygon)
                insideAreaGeometry = validGeometry->UnionCascaded();
            else {
                insideAreaGeometry = validGeometry;
            }
            insideAreaGeometry = insideAreaGeometry->Buffer(0.0);
            if (!insideAreaGeometry->IsValid()) {
                qDebug() << "Cleaned up intersection is invalid.";
                return 0;
            }
            break;
        }
        default:
            return 0;
        }
    } else {
        qDebug() << "Intersecting area with territory" << fromTerritory->territoryNumber() << "is invalid!";
        return 0;
    }

    if (insideAreaGeometry != nullptr) {
        Territory *intoTerritory = nullptr;
        QString sqlQuery;
        if (intoTerritoryId != 0) {
            // annex into existing territory
            intoTerritory = getTerritoryById(intoTerritoryId);
            if (!intoTerritory)
                return 0;
            else {
                intoTerritoryGeometry = intoTerritory->getOGRGeometry();
                if (intoTerritoryGeometry != nullptr) {
                    if (intoTerritoryGeometry->IsEmpty())
                        intoTerritoryGeometry = insideAreaGeometry->clone();
                    else
                        intoTerritoryGeometry = intoTerritoryGeometry->Union(insideAreaGeometry);
                    if (intoTerritoryGeometry == nullptr) {
                        qDebug() << "Union of area with territory" << fromTerritory->territoryNumber() << "is null!";
                        return 0;
                    }
                    if (intoTerritoryGeometry->IsEmpty()) {
                        qDebug() << "Union of area with territory" << fromTerritory->territoryNumber() << "is empty!";
                        return 0;
                    }
                    if (!intoTerritoryGeometry->IsValid()) {
                        qDebug() << "Union of area with territory" << fromTerritory->territoryNumber() << "is invalid!";
                        return 0;
                    }
                    if (intoTerritoryGeometry->getGeometryType() == wkbMultiPolygon)
                        intoTerritoryGeometry = intoTerritoryGeometry->UnionCascaded();
                } else
                    intoTerritoryGeometry = areaGeometry;
                qDebug() << "Annex into existing territory" << intoTerritory->territoryNumber();
                if (intoTerritory->setOGRGeometry(intoTerritoryGeometry))
                    territoryChanged |= intoTerritory->save();
                else
                    return 0;
            }
        } else {
            // annex into new territory
            intoTerritory = new Territory();
            intoTerritory->setTerritoryNumber(-1);
            intoTerritory->setLocality(fromTerritory->locality());
            intoTerritory->setCityId(fromTerritory->cityId());
            intoTerritory->setTypeId(fromTerritory->typeId());
            intoTerritoryGeometry = insideAreaGeometry;
            qDebug() << "Annex into new territory.";
            if (intoTerritory->setOGRGeometry(intoTerritoryGeometry)) {
                territoryChanged |= intoTerritory->save();
                intoTerritoryId = intoTerritory->territoryId();

                // copy assignments
                sqlQuery = "SELECT * FROM territory_assignment WHERE territory_id = " + QVariant(fromTerritoryId).toString() + " AND active ORDER BY checkedout_date";
                sql_items territoryAssignmentRows = sql->selectSql(sqlQuery);
                if (!territoryAssignmentRows.empty()) {
                    for (unsigned int i = 0; i < territoryAssignmentRows.size(); i++) {
                        sql_item assignmentRow = territoryAssignmentRows[i];
                        TerritoryAssignment *newTerritoryAssignment = new TerritoryAssignment(intoTerritory->territoryId());
                        newTerritoryAssignment->setPersonId(assignmentRow.value("person_id").toInt());
                        newTerritoryAssignment->setAssignedDate(assignmentRow.value("checkedout_date").toDate());
                        newTerritoryAssignment->setCompletedDate(assignmentRow.value("checkedbackin_date").toDate());
                        newTerritoryAssignment->save();
                    }
                }
            } else
                return 0;
        }

        // move street (or parts) within area to the territory
        streetChanged = moveStreetsToTerritory(fromTerritoryId, intoTerritoryId);

        // move addresses within the area to the territory
        addressChanged = moveAddressesToTerritory(fromTerritoryId, intoTerritoryId);
    }
    return (addressChanged << 2) | (streetChanged << 1) | territoryChanged;
}

bool cterritories::moveStreetsToTerritory(int fromTerritoryId, int intoTerritoryId)
{
    Territory *fromTerritory = getTerritoryById(fromTerritoryId);
    Territory *intoTerritory = getTerritoryById(intoTerritoryId);
    if (fromTerritory == nullptr || intoTerritory == nullptr || fromTerritory == intoTerritory)
        return false;
    qDebug() << "Move streets from territory" << fromTerritory->territoryNumber() << "to" << intoTerritory->territoryNumber();
    bool streetChanged = false;
    OGRSpatialReference osr;
    sql_class *sql = &Singleton<sql_class>::Instance();
    OGRGeometry *fromTerritoryGeometry = nullptr;
    fromTerritoryGeometry = fromTerritory->getOGRGeometry();
    OGRGeometry *intoTerritoryGeometry = nullptr;
    intoTerritoryGeometry = intoTerritory->getOGRGeometry();

    // move street (or parts) within area to the territory
    QString sqlQuery = "SELECT * FROM territory_street WHERE territory_id = " + QVariant(fromTerritoryId).toString() + " AND active ORDER BY street_name";
    sql_items territoryStreetRows = sql->selectSql(sqlQuery);
    if (!territoryStreetRows.empty()) {
        for (unsigned int i = 0; i < territoryStreetRows.size(); i++) {
            sql_item streetRow = territoryStreetRows[i];
            // parse WKT boundary of the street and check for intersection with territory's boundary
            QString wktGeometry = streetRow.value("wkt_geometry").toString();
            OGRGeometry *streetGeometry = nullptr;
            QByteArray bytes = wktGeometry.toUtf8();
            const char *data = bytes.constData();
            OGRErr err = OGRGeometryFactory::createFromWkt(data, &osr, &streetGeometry);
            if (err == OGRERR_NONE) {
                if (intoTerritoryGeometry->Intersects(streetGeometry)) {
                    OGRGeometry *insideAreaStreetGeometry = nullptr;
                    insideAreaStreetGeometry = intoTerritoryGeometry->Intersection(streetGeometry);
                    char *wkt_new = nullptr;
                    // check for already existing street
                    sqlQuery = QString("SELECT * FROM territory_street WHERE territory_id = %1 AND street_name = '%2' AND from_number = '%3' AND to_number = '%4' AND quantity = %5 AND streettype_id = %6 AND active ORDER BY id")
                                       .arg(QVariant(intoTerritory->territoryId()).toString())
                                       .arg(streetRow.value("street_name").toString().replace("\'", "\'\'"))
                                       .arg(streetRow.value("from_number").toString())
                                       .arg(streetRow.value("to_number").toString())
                                       .arg(streetRow.value("quantity").toString())
                                       .arg(streetRow.value("streettype_id").toString());
                    sql_items existingTerritoryStreetRows = sql->selectSql(sqlQuery);
                    if (existingTerritoryStreetRows.size() > 0) {
                        // add geometry to existing street
                        wktGeometry = existingTerritoryStreetRows[0].value("wkt_geometry").toString();
                        OGRGeometry *existingStreetGeometry = nullptr;
                        bytes = wktGeometry.toUtf8();
                        data = bytes.constData();
                        OGRErr err = OGRGeometryFactory::createFromWkt(data, &osr, &existingStreetGeometry);
                        if (err == OGRERR_NONE) {
                            insideAreaStreetGeometry = existingStreetGeometry->Union(insideAreaStreetGeometry);
                            insideAreaStreetGeometry = OGRGeometryFactory::forceToLineString(insideAreaStreetGeometry);
                            insideAreaStreetGeometry->exportToWkt(&wkt_new);

                            sql_item existingStreetItem;
                            int existingStreetId = existingTerritoryStreetRows[0].value("id").toInt();
                            existingStreetItem.insert("wkt_geometry", QString::fromStdString(wkt_new));
                            if (sql->updateSql("territory_street", "id", QString::number(existingStreetId), &existingStreetItem)) {
                                streetChanged = true;
                            }
                        }
                    } else {
                        // add new street
                        insideAreaStreetGeometry->exportToWkt(&wkt_new);
                        TerritoryStreet *newTerritoryStreet = new TerritoryStreet(
                                intoTerritory->territoryId(),
                                streetRow.value("street_name").toString(),
                                streetRow.value("from_number").toString(),
                                streetRow.value("to_number").toString(),
                                streetRow.value("quantity").toInt(),
                                streetRow.value("streettype_id").toInt(),
                                QString(wkt_new));
                        newTerritoryStreet->setIsDirty(true);
                        streetChanged |= newTerritoryStreet->save();
                    }

                    // update the remaining part of the street within the original territory boundary
                    OGRGeometry *outsideAreaStreetGeometry = nullptr;
                    outsideAreaStreetGeometry = fromTerritoryGeometry->Intersection(streetGeometry);
                    outsideAreaStreetGeometry->exportToWkt(&wkt_new);

                    sql_item s;
                    int streetId = streetRow.value("id").toInt();
                    s.insert("wkt_geometry", QString::fromStdString(wkt_new));
                    s.insert("active", !outsideAreaStreetGeometry->IsEmpty());
                    streetChanged |= sql->updateSql("territory_street", "id", QString::number(streetId), &s);
                }
            }
        }
    }
    return streetChanged;
}

bool cterritories::moveAddressesToTerritory(int fromTerritoryId, int intoTerritoryId)
{
    bool addressChanged = false;
    sql_class *sql = &Singleton<sql_class>::Instance();
    Territory *intoTerritory = getTerritoryById(intoTerritoryId);
    if (intoTerritory == nullptr)
        return false;
    OGRGeometry *intoTerritoryGeometry = nullptr;
    intoTerritoryGeometry = intoTerritory->getOGRGeometry();

    // move addresses within the area to the territory
    QString sqlQuery = "SELECT * FROM territory_address WHERE territory_id = " + QVariant(fromTerritoryId).toString() + " AND active ORDER BY street, housenumber";
    sql_items territoryAddressRows = sql->selectSql(sqlQuery);
    if (!territoryAddressRows.empty()) {
        for (unsigned int i = 0; i < territoryAddressRows.size(); i++) {
            sql_item addressRow = territoryAddressRows[i];

            QRegularExpressionMatch match = coordinateRegExp->match(addressRow.value("wkt_geometry").toString());
            if (match.hasMatch()) {
                double lon = match.captured(1).toDouble();
                double lat = match.captured(2).toDouble();
                OGRPoint *point = new OGRPoint(lon, lat);

                if (point->Within(intoTerritoryGeometry)) {
                    sql_item s;
                    int addressId = addressRow.value("id").toInt();
                    s.insert("territory_id", intoTerritory->territoryId());
                    if (sql->updateSql("territory_address", "id", QString::number(addressId), &s)) {
                        addressChanged = true;
                    }
                }
            }
        }
    }
    return addressChanged;
}
#endif

CSVSchema *cterritories::getCSVschema(QUrl fileUrl, QString delimiter)
{
    CSVSchema *schema = new CSVSchema();

    QFile csvFile(fileUrl.toLocalFile());
    if (!csvFile.open(QIODevice::ReadOnly)) {
        qDebug() << "Could not open " << fileUrl.toLocalFile();
        return schema;
    }
    QString headerLine;
    QTextStream in(&csvFile);
    if (!in.atEnd()) {
        headerLine = in.readLine();
    }
    csvFile.close();

    if (!headerLine.isEmpty()) {
        QList<QString> delimiters = { ",", "\t", ";", "|", ":" };
        if (delimiter != "")
            delimiters = { delimiter };
        for (int i = 0; i < delimiters.length(); i++) {
            QList<QString> fields = headerLine.split(delimiters[i]);
            if (fields.count() > 1) {
                schema->setDelimiter(delimiters[i]);
                schema->setFields(fields);
                return schema;
            }
        }
    }

    return schema;
}

/**
 * @brief importAddresses - Import territory addresses from csv-file
 * @param fileUrl - file name
 * @param addressField, nameField - field index in csv file
 * @param delimiter - csv delimiter
 * @param territoryId - territory to import into
 * @param addressTypeId - type of addresses
 * @param failedFileUrl - output filename for failed addresses
 * @return - number of imported addresses; negative number = failure
 */
int cterritories::importAddresses(QUrl fileUrl, int addressField, int nameField,
                                  QString delimiter, int territoryId, int addressTypeNumber, QUrl failedFileUrl)
{
    if (territoryId < 1)
        return -2;

    QFile csvFile(fileUrl.toLocalFile());
    if (!csvFile.open(QIODevice::ReadOnly | QIODevice::Text))
        return -1;

    // skip header
    QString header;
    if (!csvFile.atEnd())
        header = csvFile.readLine();

    sql_class *sql;
    sql = &Singleton<sql_class>::Instance();
    sql->startTransaction();

    int rowCount = 0;
    int importCount = 0;

    QString failedAddresses;

    while (!csvFile.atEnd()) {
        rowCount += 1;
        QString line = csvFile.readLine();
        QStringList values = line.split(delimiter);
        if (addressField >= values.size())
            break;
        QString address = values[addressField];
        QString addressName = values[nameField].trimmed();

        bool importFailed = false;
        for (int i = 0; i < 5; i++) {
            QNetworkAccessManager manager;
            QEventLoop q;
            QTimer tT;

            tT.setSingleShot(true);
            connect(&tT, SIGNAL(timeout()), &q, SLOT(quit()));
            connect(&manager, SIGNAL(finished(QNetworkReply *)),
                    &q, SLOT(quit()));

            QNetworkRequest request;
            request.setUrl(getGeocodeUrl(address));
            QNetworkReply *reply = manager.get(request);

            tT.start(5000); // 5s timeout
            q.exec();

            if (tT.isActive()) {
                // download complete
                tT.stop();

                QJsonDocument doc = QJsonDocument::fromJson(reply->readAll());
                QList<QGeoLocation *> geocodeResultList = getGeocodeResults(doc);

                if (geocodeResultList.length() == 1) {
                    QGeoAddress geoAddress = geocodeResultList[0]->address();
                    QGeoCoordinate geoCoordinate = geocodeResultList[0]->coordinate();
                    QVariantMap extAttributes = geocodeResultList[0]->extendedAttributes();
                    QString wktGeometry = extAttributes.value("wktGeometry").toString();
                    if (wktGeometry.isEmpty()) {
                        wktGeometry = "POINT(" + QVariant(geoCoordinate.longitude()).toString() + " " + QVariant(geoCoordinate.latitude()).toString() + ")";
                    }
                    TerritoryAddress *ta =
                            new TerritoryAddress(territoryId,
                                                 geocodeResultList[0]->address().country(),
                                                 geocodeResultList[0]->address().state(),
                                                 geocodeResultList[0]->address().county(),
                                                 geocodeResultList[0]->address().city(),
                                                 geocodeResultList[0]->address().district(),
                                                 geocodeResultList[0]->address().street(),
                                                 geocodeResultList[0]->address().streetNumber(),
                                                 geocodeResultList[0]->address().postalCode(),
                                                 wktGeometry);

                    if (addressName.startsWith("\""))
                        addressName.remove(0, 1);
                    if (addressName.endsWith("\""))
                        addressName.remove(addressName.size() - 1, 1);
                    ta->setName(addressName);
                    ta->setAddressTypeNumber(addressTypeNumber);

                    if (ta->save()) {
                        importFailed = false;
                        importCount += 1;
                        break;
                    }
                } else
                    importFailed = true;

            } else {
                // timeout
                importFailed = true;
            }
        }

        if (importFailed)
            failedAddresses += line;

        emit addressImportProgressChanged(rowCount, importCount);
    }

    sql->commitTransaction();

    // output failed addresses
    if (!failedAddresses.isEmpty() && failedFileUrl.isValid()) {
        QFile outputfile(failedFileUrl.toLocalFile());
        if (QFile::exists(failedFileUrl.toLocalFile()))
            QFile::remove(failedFileUrl.toLocalFile());
        if (!outputfile.open(QIODevice::WriteOnly | QIODevice::Text))
            QMessageBox::warning(nullptr,
                                 tr("Save failed addresses", "Territory address import"),
                                 tr("The file is in read only mode", "Save failed addresses in territory data import"));
        else {
            QTextStream out(&outputfile);
            out << header << failedAddresses;
            outputfile.close();
        }
    }

    if (importCount > 0)
        emit addressGeometryChanged();

    return importCount;
}

#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
// GDAL missing
#elif defined(Q_OS_LINUX) || defined(Q_OS_MAC) || defined(Q_OS_WIN)
void cterritories::requestStreetList(int territoryId)
{
    Territory *t = getTerritoryById(territoryId);
    QString wktGeometry = t->wktGeometry();
    QString latLonList;

    // use bounding box, since polygon may result in too long http request
    OGRSpatialReference osr;
    OGRGeometry *territoryGeometry = nullptr;
    QByteArray bytes = wktGeometry.toUtf8();
    const char *data = bytes.constData();
    OGRErr err = OGRGeometryFactory::createFromWkt(data, &osr, &territoryGeometry);
    if (err != OGRERR_NONE) {
        // process error, like emit signal
    } else {
        OGREnvelope territoryEnvelope;
        territoryGeometry->getEnvelope(&territoryEnvelope);
        latLonList = QString("%1 %2 %1 %4 %3 %4 %3 %2 %1 %2")
                             .arg(territoryEnvelope.MinY)
                             .arg(territoryEnvelope.MinX)
                             .arg(territoryEnvelope.MaxY)
                             .arg(territoryEnvelope.MaxX);
    }
    if (latLonList != "") {
        QString overpassAPIUrl = "http://overpass-api.de/api/interpreter?data=";
        QString osmQLString = "[out:json][timeout:25];"
                              "way(poly:\""
                + latLonList + "\")[highway][name];"
                               "out geom;";
        QNetworkRequest request;
        request.setUrl(overpassAPIUrl + osmQLString);
        QNetworkAccessManager *streetRequest;
        streetRequest = new QNetworkAccessManager();
        connect(streetRequest, SIGNAL(finished(QNetworkReply *)),
                SLOT(streetRequestFinished(QNetworkReply *)));
        QNetworkReply *reply = streetRequest->get(request);
        reply->setProperty("territoryId", territoryId);
    }
}
#endif

void cterritories::getTerritoryStreets(int territoryId)
{
    AccessControl *ac = &Singleton<AccessControl>::Instance();
    if (!ac->user() || !ac->user()->hasPermission(PermissionRule::CanViewTerritories))
        return;

    sql_class *sql = &Singleton<sql_class>::Instance();
    QString sqlQuery = QString("SELECT s.id, s.territory_id, s.wkt_geometry, s.streettype_id FROM territorystreets s "
                               "JOIN territories t ON s.territory_id = t.id "
                               "WHERE (%1 OR t.person_id = %2) AND "
                               "s.wkt_geometry IS NOT NULL AND s.wkt_geometry <> ''")
                               .arg(!ac->isActive() || ac->user()->hasPermission(PermissionRule::CanViewTerritoryAddresses))
                               .arg(ac->user()->personId());
    sql_items streets = sql->selectSql(sqlQuery);

    QVariantMap streetMap;
    if (!streets.empty()) {
        for (unsigned int i = 0; i < streets.size(); i++) {
            const sql_item street = streets[i];
            int street_id = street.value("id").toInt();
            int territory_id = street.value("territory_id").toInt();
            int streettype_id = street.value("streettype_id").toInt();
            QString input = street.value("wkt_geometry").toString();
            // catch MULTILINESTRING and LINESTRING
            QRegularExpressionMatchIterator iMultiLineString = multiLineStringRegExp->globalMatch(input);

            int iLine = 0;
            while (iMultiLineString.hasNext()) {
                QRegularExpressionMatch lineStringMatch = iMultiLineString.next();
                QString lineString = lineStringMatch.captured(0);
                Street street;
                QGeoPath geoPath;

                QRegularExpressionMatchIterator iMatch = coordinateRegExp->globalMatch(lineString);
                while (iMatch.hasNext()) {
                    QRegularExpressionMatch match = iMatch.next();
                    double longitude = match.captured(1).toDouble();
                    double latitude = match.captured(2).toDouble();
                    QGeoCoordinate coord(latitude, longitude);
                    geoPath.addCoordinate(coord);
                }
                street.setStreetId(street_id);
                street.setTerritoryId(territory_id);
                street.setPath(geoPath);
                street.setStreetTypeId(streettype_id);

                streetMap[QVariant(street_id).toString() + "_" + QVariant(iLine).toString()] = QVariant::fromValue(street); // qVariantFromValue(street);
                iLine += 1;
            }
        }
    }
    emit streetsLoaded(streetMap, territoryId);
}

/**
 * @brief getClosestPoint - Returns the closest from one of the territory boundaries' points to the given point.
 * @param point - center point
 * @param tolerance - search radius
 */
QGeoCoordinate cterritories::getClosestPoint(QGeoCoordinate point, double tolerance, QList<QGeoCoordinate> coordinates)
{
    QGeoCoordinate closestPoint;
    if (coordinates.count() > 0) {
        double distance(tolerance);
        for (int i = 0; i < coordinates.count(); ++i) {
            double newDistance(point.distanceTo(coordinates[i]));
            if (newDistance < distance) {
                distance = newDistance;
                closestPoint = coordinates[i];
            }
        }
    }
    return closestPoint;
}
