#ifndef STORAGESERVICE_H
#define STORAGESERVICE_H

#include <QUrl>
#include <QFile>
#include <QString>
#include <QDebug>
#include <QObjectBindableProperty>
#include <QNetworkRequestFactory>
#include <QRestReply>
#include <QRestAccessManager>
#include <QOAuth2AuthorizationCodeFlow>
#include <QOAuthHttpServerReplyHandler>
#include <QOAuthUriSchemeReplyHandler>
#include <QNetworkReply>
#include <QJsonDocument>
#include <QJsonObject>
#include <QJsonArray>
#include <QUrlQuery>
#include <QMessageBox>
#include <QSettings>
#include <QEventLoop>
#include <QStandardItemModel>
#include <QCryptographicHash>
#include <QtQml/qqml.h>

QT_FORWARD_DECLARE_CLASS(QRestAccessManager)

// Expose enum from Qt to QML, and protect enum for proper use in C++
class StorageServiceProviderClass
{
    Q_GADGET

public:
    enum class Provider {
        Dropbox = 0,
        Nextcloud = 1
    };
    Q_ENUM(Provider)

    Q_INVOKABLE static QString toString(Provider provider)
    {
        QMetaEnum metaEnum = QMetaEnum::fromType<Provider>();
        return metaEnum.valueToKey(int(provider));
    }

private:
    explicit StorageServiceProviderClass();
};
typedef StorageServiceProviderClass::Provider StorageServiceProvider;

// Base class
class StorageAccount : public QObject
{
    Q_OBJECT
    Q_PROPERTY(QString loginName READ loginName WRITE setLoginName BINDABLE bindableLoginName)
    Q_PROPERTY(QString name READ name WRITE setName BINDABLE bindableName)
    Q_PROPERTY(QString email READ email WRITE setEmail BINDABLE bindableEmail)
    Q_PROPERTY(QString syncFile READ syncFile WRITE setSyncFile BINDABLE bindableSyncFile)
public:
    StorageAccount(StorageServiceProvider provider, QObject *parent = nullptr)
        : QObject(parent), m_provider(provider) { }
    StorageAccount(StorageServiceProvider provider, QString name, QString email, QObject *parent = nullptr)
        : QObject(parent), m_provider(provider), b_name(name), b_email(email) { }
    StorageAccount(StorageServiceProvider provider, QString loginName, QString name, QString email, QObject *parent = nullptr)
        : QObject(parent), m_provider(provider), b_loginName(loginName), b_name(name), b_email(email) { }

    QString loginName() { return b_loginName.value(); }
    void setLoginName(QString newValue) { b_loginName = newValue; }
    QBindable<QString> bindableLoginName() { return &b_loginName; }

    QString name() { return b_name.value(); }
    void setName(QString newValue) { b_name = newValue; }
    QBindable<QString> bindableName() { return &b_name; }

    QString email() { return b_email.value(); }
    void setEmail(QString newValue) { b_email = newValue; }
    QBindable<QString> bindableEmail() { return &b_email; }

    QString syncFile() { return b_syncFile.value(); }
    void setSyncFile(QString newValue)
    {
        QSettings s;
        QString settingsKey = getSettingsKey("syncfile");
        if (newValue.compare(s.value(settingsKey, "").toString()) != 0)
            s.setValue(settingsKey, newValue);
        b_syncFile = newValue;
    }
    QBindable<QString> bindableSyncFile() { return &b_syncFile; }

    void reset()
    {
        setLoginName("");
        setName("");
        setEmail("");
        setSyncFile("");
    }

signals:
    void loginNameChanged();
    void nameChanged();
    void emailChanged();
    void syncFileChanged();

private:
    StorageServiceProvider m_provider;

    QString getSettingsKey(QString key)
    {
        return QString("%1/%2").arg(StorageServiceProviderClass::toString(m_provider)).arg(key).toLower();
    }

protected:
    Q_OBJECT_BINDABLE_PROPERTY(StorageAccount, QString, b_loginName, &StorageAccount::loginNameChanged);
    Q_OBJECT_BINDABLE_PROPERTY(StorageAccount, QString, b_name, &StorageAccount::nameChanged);
    Q_OBJECT_BINDABLE_PROPERTY(StorageAccount, QString, b_email, &StorageAccount::emailChanged);
    Q_OBJECT_BINDABLE_PROPERTY(StorageAccount, QString, b_syncFile, &StorageAccount::syncFileChanged);
};

struct StorageUser {
public:
    QString displayName;
    QString email;
    bool owner;
};

class StorageService : public QObject
{
    Q_OBJECT
    QML_ELEMENT
    Q_PROPERTY(QString providerName READ providerName BINDABLE bindableProviderName)
    Q_PROPERTY(StorageAccount *account READ account WRITE setAccount BINDABLE bindableAccount)
    Q_PROPERTY(bool isAuthenticated READ isAuthenticated BINDABLE bindableIsAuthenticated)
    Q_PROPERTY(QString errorString READ errorString BINDABLE bindableErrorString)

public:
    StorageService(StorageServiceProvider provider, QObject *parent = nullptr);

    QString providerName() { return b_providerName.value(); }
    QBindable<QString> bindableProviderName() { return &b_providerName; }

    QPointer<StorageAccount> account() { return b_account.value(); }
    void setAccount(QPointer<StorageAccount> newValue) { b_account = newValue; }
    QBindable<QPointer<StorageAccount>> bindableAccount() { return &b_account; }

    bool isAuthenticated() { return b_isAuthenticated.value(); }
    void setIsAuthenticated(bool newValue) { b_isAuthenticated = newValue; }
    QBindable<bool> bindableIsAuthenticated() { return &b_isAuthenticated; }

    QString errorString() { return b_errorString.value(); }
    QBindable<QString> bindableErrorString() { return &b_errorString; }

    virtual void authenticate() {};
    virtual void revoke() {};

    virtual void loadAccount() {};

    virtual QDateTime upload(QString /*localFileName*/, QString /*relativeCloudFileName*/) { return QDateTime(); };
    virtual QDateTime upload(QByteArray /*data*/, QString /*relativeCloudFileName*/) { return QDateTime(); };
    virtual void download(QString /*relativeCloudFileName*/, QString /*localFileName*/, QDateTime & /*modifiedDate*/) {};
    virtual void download(QString /*relativeCloudFileName*/, QByteArray & /*content*/, QDateTime & /*modifiedDate*/) {};

    virtual QDateTime getModifiedDate(QString /*cloudFileName*/) { return QDateTime(); };
    virtual bool deleteFile(QString /*cloudFileName*/) { return false; };

    virtual bool createFolder(QString /*path*/) { return false; };
    virtual QString getSharedLink(QString /*path*/) { return ""; };

    virtual bool fileExists(QString /*path*/) { return false; };

    virtual QAbstractItemModel *searchFile(QString /*query*/) { return nullptr; };
    virtual QList<StorageUser> listFolderMembers(QString /*path*/) { return QList<StorageUser>(); };

    virtual bool getAccessToken(QString /*code*/) { return false; };

signals:
    void providerNameChanged();
    void accountChanged();
    void isAuthenticatedChanged();
    void authorizeWithBrowserStarted();
    void statusChanged(QAbstractOAuth::Status status);
    void errorStringChanged();

private:
    StorageServiceProvider m_provider;
    void setProviderName(QString newValue) { b_providerName = newValue; }
    void setErrorString(QString newValue) { b_errorString = newValue; }

protected:
    Q_OBJECT_BINDABLE_PROPERTY(StorageService, QString, b_providerName, &StorageService::providerNameChanged);
    Q_OBJECT_BINDABLE_PROPERTY(StorageService, QPointer<StorageAccount>, b_account, &StorageService::accountChanged);
    Q_OBJECT_BINDABLE_PROPERTY(StorageService, bool, b_isAuthenticated, &StorageService::isAuthenticatedChanged);
    Q_OBJECT_BINDABLE_PROPERTY(StorageService, QString, b_errorString, &StorageService::errorStringChanged);
};

#endif // STORAGESERVICE_H
