#include "nextcloud.h"
#include "qelapsedtimer.h"
#include "qrestreply.h"
#include "qtextdocument.h"
#include "qthread.h"
#include "qtimer.h"
#include <QDesktopServices>

using namespace Qt::StringLiterals;

static const auto displayNameTag = u"d:displayname"_s;
static const auto fileIdTag = u"oc:fileid"_s;
static const auto getLastModifiedTag = u"d:getlastmodified"_s;
static const auto hrefTag = u"d:href"_s;
static const auto ownerIdTag = u"oc:owner-id"_s;
static const auto ownerDisplayNameTag = u"oc:owner-display-name"_s;
static const auto propTag = u"d:prop"_s;
static const auto propStatTag = u"d:propstat"_s;
static const auto responseTag = u"d:response"_s;
static const auto shareesTag = u"nc:sharees"_s;
static const auto statusTag = u"d:status"_s;

Nextcloud::Nextcloud(QString baseUrl, QObject *parent)
    : StorageService(StorageServiceProvider::Nextcloud, parent)
{
    ncAuth = new QOAuth2AuthorizationCodeFlow(this);
    connect(ncAuth, &QOAuth2AuthorizationCodeFlow::tokenChanged, this, [=](const QString token) {
        setIsAuthenticated(!token.isEmpty());
    });
    QNetworkAccessManager *qnam = new QNetworkAccessManager(this);
    network = new QRestAccessManager(qnam, qnam);

    apiBaseUrl = baseUrl;
    ncApi.setBaseUrl(QUrl(apiBaseUrl));

    accessTokenUrl = QString("%1/index.php/login/v2").arg(apiBaseUrl);
    ncAuth->setTokenUrl(QUrl(accessTokenUrl));
    ncAuth->setPkceMethod(QOAuth2AuthorizationCodeFlow::PkceMethod::None);
    ncAuth->setModifyParametersFunction([&](QAbstractOAuth::Stage stage, QMultiMap<QString, QVariant> *params) {
        Q_UNUSED(params)
        if (stage == QAbstractOAuth::Stage::RequestingAuthorization) {
            qDebug() << "Nextcloud-API modify parameters function (RequestingAuthorization)";
        } else if (stage == QAbstractOAuth::Stage::RefreshingAccessToken) {
            qDebug() << "Nextcloud-API modify parameters function (RefreshingAccessToken)";
        } else if (stage == QAbstractOAuth::Stage::RequestingAccessToken) {
            qDebug() << "Nextcloud-API modify parameters function (RequestingAccessToken)";
        }
    });

    // load previous login settings
    QString token = settings.value("nextcloud/token", "").toString();
    QList<QByteArray> tokenParts = QByteArray::fromBase64(token.toUtf8(), QByteArray::Base64Encoding).split(':');
    if (tokenParts.count() == 2) {
        QString email = settings.value("nextcloud/email", "").toString();
        QString name = settings.value("nextcloud/name", "").toString();
        setAccount(new StorageAccount(StorageServiceProvider::Nextcloud, tokenParts[0], name, email, this)); // set login name
        account()->setSyncFile(settings.value("nextcloud/syncfile", "").toString());
        ncAuth->setToken(token);
    }

    QString refreshToken = settings.value("nextcloud/refreshtoken").toString();
    ncAuth->setRefreshToken(refreshToken);
    tokenExpiration = settings.value("nextcloud/expirationAt").toDateTime();

    connect(ncAuth, &QAbstractOAuth::granted, this, [=]() {
        qDebug() << "Nextcloud-API granted";
        ncApi.setBearerToken(ncAuth->token().toLatin1());
        saveSettings(QJsonDocument());
#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
        uriSchemeReplyHandler->close();
#endif
    });
    connect(ncAuth, &QOAuth2AuthorizationCodeFlow::authorizeWithBrowser, this, &QDesktopServices::openUrl);
    // provide a signal e.g. to close the Dropbox panel
    connect(ncAuth, &QOAuth2AuthorizationCodeFlow::authorizeWithBrowser, this, &Nextcloud::authorizeWithBrowserStarted);
    connect(ncAuth, &QOAuth2AuthorizationCodeFlow::statusChanged, this, &Nextcloud::statusChanged);
    // OAuth Error Handling
    connect(ncAuth, &QOAuth2AuthorizationCodeFlow::serverReportedErrorOccurred, [](const QString &error, const QString &errorDescription, const QUrl &uri) {
        qDebug() << "OAuth Error:" << error;
        qDebug() << "Error Description:" << errorDescription;
        qDebug() << "URI:" << uri;
    });
}

bool Nextcloud::initLoginFlow()
{
    qDebug() << "Initiate a login";

    QNetworkRequest req(ncAuth->tokenUrl());
    req.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");

    QNetworkReply *networkReply = network->post(req, QByteArray());
    connect(networkReply, &QNetworkReply::sslErrors, this, [=]() {
        qDebug() << "WARNING: The connection to the Nextcloud server is not secure!";
        networkReply->ignoreSslErrors();
    });

    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        QByteArray replyBody = restReply.readBody();
        QJsonParseError parseError;
        if (auto doc = QJsonDocument::fromJson(replyBody, &parseError); parseError.error == QJsonParseError::NoError && doc.isObject()) {
            QString token;
            QString endpoint;
            QString login;
            if (doc.object().contains("poll")) {
                token = doc.object().value("poll").toObject().value("token").toString();
                endpoint = doc.object().value("poll").toObject().value("endpoint").toString();
            }
            if (doc.object().contains("login")) {
                login = doc.object().value("login").toString();
                ncAuth->setAuthorizationUrl(QUrl(login));
            }

            // 2. url in login should be opened in the default browser
            ncAuth->grant();

            // 3. The program should directly start polling the poll endpoint
            QString loginName;
            QString appPassword;
            QElapsedTimer timer;
            timer.start();
            req = QNetworkRequest(QUrl(endpoint));
            req.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
            QString postData = QString("token=%1").arg(token);
            // the token is valid for 20 minutes
            while (!timer.hasExpired(1000 * 60 * 20) && appPassword.isNull()) {
                networkReply = network->post(req, postData.toUtf8());
                connect(networkReply, &QNetworkReply::sslErrors, this, [=]() {
                    qDebug() << "WARNING: The connection to the Nextcloud server is not secure!";
                    networkReply->ignoreSslErrors();
                });

                QEventLoop loop;
                connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
                loop.exec();

                QRestReply restReply(networkReply);
                if (restReply.isSuccess()) {
                    if (auto doc = restReply.readJson(); doc && doc->isObject()) {
                        apiBaseUrl = doc->object().value("server").toString();
                        loginName = doc->object().value("loginName").toString();
                        setAccount(new StorageAccount(StorageServiceProvider::Nextcloud, loginName, "", "", this));
                        appPassword = doc->object().value("appPassword").toString();
                    }
                } else {
                    qDebug() << readReply(restReply, "acquire app password");
                }
                QThread::sleep(1);
            }

            qDebug() << "server:" << apiBaseUrl;
            ncApi.setBaseUrl(QUrl(apiBaseUrl));

            qDebug() << "appPassword:" << appPassword;

            auto base64Encode = [](QByteArray value) {
                value = value.toBase64();
                return value.replace("+", "-").replace("/", "_").replace("=", "");
            };

            settings.setValue("nextcloud/server", apiBaseUrl);
            QString strValue = QString("%1:%2").arg(loginName).arg(appPassword);
            qDebug() << "Server user and password:" << strValue;
            setToken(base64Encode(strValue.toUtf8()));
        } else {
            qDebug() << replyBody;
        }
    } else {
        qDebug() << readReply(restReply, "initiate login");
    }
    return false;
}

void Nextcloud::authenticate()
{
    qDebug() << "authenticate";

    if (!ncAuth->token().isEmpty()) {
        qDebug() << "Already authenticated!";
        return;
    }
    initLoginFlow();

    // Initiate the authorization
    // #if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
    //     if (uriSchemeReplyHandler->listen()) {
    //         ncAuth->grant();
    //     }
    // #else
    //     ncAuth->grant();
    // #endif
}

void Nextcloud::revoke()
{
    qDebug() << "revoke";
    if (ncAuth->token().isEmpty())
        return;
    initCall();

    QNetworkRequest req = ncApi.createRequest("ocs/v2.php/core/apppassword");
    QHttpHeaders headers;
    headers.append("OCS-APIRequest", "true");
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Basic "_s + ncAuth->token());
    req.setHeaders(headers);

    QNetworkReply *networkReply = network->deleteResource(req);
    connect(networkReply, &QNetworkReply::sslErrors, this, [=]() {
        qDebug() << "WARNING: The connection to the Nextcloud server is not secure!";
        networkReply->ignoreSslErrors();
    });
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        qDebug() << "revoke finished successfully";
    } else {
        qDebug() << readReply(restReply, "revoke token");
    }

    // clear settings
    settings.setValue("nextcloud/refreshtoken", "");
    settings.setValue("nextcloud/name", "");
    settings.setValue("nextcloud/email", "");
    settings.setValue("nextcloud/syncfile", "");
    account()->reset();
    // clear token and trigger authorized changed signal
    this->setToken("");
}

void Nextcloud::setToken(const QString token)
{
    qDebug() << "setToken" << token;
    settings.setValue("nextcloud/token", token);
    ncAuth->setToken(token);
    // QAbstractOAuth::Status newstatus = token.isEmpty() ? QAbstractOAuth::Status::NotAuthenticated : QAbstractOAuth::Status::Granted;
    // if (ncAuth->status() != newstatus) {
    //     qDebug() << "status:" << int(ncAuth->status());
    //     emit ncAuth->statusChanged(newstatus);
    //     qDebug() << "status is:" << int(ncAuth->status()) << "; should be:" << int(newstatus);
    // }
}

QString Nextcloud::errorString()
{
    return m_error;
}

QDateTime Nextcloud::upload(QString localFileName, QString relativeCloudFileName)
{
    QFile file(localFileName);
    if (!file.open(QIODevice::ReadOnly))
        return QDateTime();
    QByteArray postData = file.readAll();
    file.close();

    return upload(postData, relativeCloudFileName);
}

QDateTime Nextcloud::upload(QByteArray data, QString relativeCloudFileName)
{
    initCall();

    QString folder = QUrl(relativeCloudFileName).toString(QUrl::RemoveFilename | QUrl::NormalizePathSegments | QUrl::StripTrailingSlash);
    createFolder(folder);

    QNetworkRequest req = ncApi.createRequest(QString("remote.php/dav/files/%1%2%3")
                                                      .arg(account()->loginName())
                                                      .arg(relativeCloudFileName.startsWith("/") ? "" : "/")
                                                      .arg(relativeCloudFileName));
    QHttpHeaders headers;
    headers.append("OCS-APIRequest", "true");
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Basic "_s + ncAuth->token());
    headers.append(QHttpHeaders::WellKnownHeader::ContentType, "application/octet-stream");
    req.setHeaders(headers);

    QNetworkReply *networkReply = network->put(req, data);
    connect(networkReply, &QNetworkReply::sslErrors, this, [=]() {
        qDebug() << "WARNING: The connection to the Nextcloud server is not secure!";
        networkReply->ignoreSslErrors();
    });
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QDateTime dt;
    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        // request modified time
        dt = getModifiedDate(relativeCloudFileName); // QDateTime::currentDateTime();
    } else {
        qDebug() << readReply(restReply, "upload file");
    }
    return dt;
}

void Nextcloud::download(QString relativeCloudFileName, QString localFileName, QDateTime &modifiedDate)
{
    QByteArray data;
    download(relativeCloudFileName, data, modifiedDate);

    QFile file(localFileName);
    if (file.open(QIODevice::WriteOnly)) {
        file.write(data);
        file.close();
    }
}

void Nextcloud::download(QString relativeCloudFileName, QByteArray &content, QDateTime &modifiedDate)
{
    if (relativeCloudFileName.isEmpty())
        return;
    initCall();

    QNetworkRequest req = ncApi.createRequest(QString("remote.php/dav/files/%1%2%3")
                                                      .arg(account()->loginName())
                                                      .arg(relativeCloudFileName.startsWith("/") ? "" : "/")
                                                      .arg(relativeCloudFileName));
    QHttpHeaders headers;
    headers.append("OCS-APIRequest", "true");
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Basic "_s + ncAuth->token());
    headers.append(QHttpHeaders::WellKnownHeader::ContentType, "application/octet-stream");
    req.setHeaders(headers);

    QNetworkReply *networkReply = network->get(req, QByteArray());
    connect(networkReply, &QNetworkReply::sslErrors, this, [=]() {
        qDebug() << "WARNING: The connection to the Nextcloud server is not secure!";
        networkReply->ignoreSslErrors();
    });
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QDateTime dt;
    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        // read file content
        content = restReply.readBody();
        // request modified time
        modifiedDate = getModifiedDate(relativeCloudFileName);
    } else {
        qDebug() << readReply(restReply, "download file");
    }
}

QDateTime Nextcloud::getModifiedDate(QString cloudFileName)
{
    if (cloudFileName.isEmpty())
        return QDateTime();

    initCall();

    QNetworkRequest req = ncApi.createRequest(QString("remote.php/dav/files/%1%2%3")
                                                      .arg(account()->loginName())
                                                      .arg(cloudFileName.startsWith("/") ? "" : "/")
                                                      .arg(cloudFileName));
    QHttpHeaders headers;
    headers.append("OCS-APIRequest", "true");
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Basic "_s + ncAuth->token());
    req.setHeaders(headers);

    QByteArray postData = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>"
                          "<d:propfind xmlns:d=\"DAV:\" xmlns:oc=\"http://owncloud.org/ns\" xmlns:nc=\"http://nextcloud.org/ns\">"
                          "  <d:prop>"
                          "    <d:getlastmodified/>"
                          "    <d:getcontentlength/>"
                          "    <d:getcontenttype/>"
                          "    <oc:permissions/>"
                          "    <d:resourcetype/>"
                          "    <d:getetag/>"
                          "  </d:prop>"
                          "</d:propfind>";
    QNetworkReply *networkReply = network->sendCustomRequest(req, "PROPFIND", postData);
    connect(networkReply, &QNetworkReply::sslErrors, this, [=]() {
        qDebug() << "WARNING: The connection to the Nextcloud server is not secure!";
        networkReply->ignoreSslErrors();
    });
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        QDomDocument doc("xml");
        doc.setContent(restReply.readBody());
        QDomNode responseElement = doc.documentElement().firstChild();
        QDomElement lastmodifiedElement = responseElement.firstChildElement(propStatTag)
                                                  .firstChildElement(propTag)
                                                  .firstChildElement(getLastModifiedTag);
        QDateTime dt = QDateTime::fromString(lastmodifiedElement.text().replace("GMT", "+0000"), Qt::RFC2822Date);
        return dt;
    } else {
        qDebug() << readReply(restReply, "get file metadata");
    }
    return QDateTime();
}

bool Nextcloud::deleteFile(QString cloudFileName)
{
    initCall();

    QNetworkRequest req = ncApi.createRequest(QString("remote.php/dav/files/%1%2%3")
                                                      .arg(account()->loginName())
                                                      .arg(cloudFileName.startsWith("/") ? "" : "/")
                                                      .arg(cloudFileName));
    QHttpHeaders headers;
    headers.append("OCS-APIRequest", "true");
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Basic "_s + ncAuth->token());
    req.setHeaders(headers);

    QNetworkReply *networkReply = network->post(req, QByteArray());
    connect(networkReply, &QNetworkReply::sslErrors, this, [=]() {
        qDebug() << "WARNING: The connection to the Nextcloud server is not secure!";
        networkReply->ignoreSslErrors();
    });
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        return true;
    } else {
        qDebug() << readReply(restReply, "delete file");
    }
    return false;
}

bool Nextcloud::createFolder(QString path)
{
    QString parentFolder = QUrl(path).toString(QUrl::RemoveFilename | QUrl::NormalizePathSegments | QUrl::StripTrailingSlash);
    if (!parentFolder.replace("/", "").isEmpty()) {
        if (!fileExists(parentFolder))
            createFolder(parentFolder);
    } else
        initCall();

    QNetworkRequest req = ncApi.createRequest(QString("remote.php/dav/files/%1%2%3")
                                                      .arg(account()->loginName())
                                                      .arg(path.startsWith("/") ? "" : "/")
                                                      .arg(path));
    QHttpHeaders headers;
    headers.append("OCS-APIRequest", "true");
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Basic "_s + ncAuth->token());
    req.setHeaders(headers);

    QNetworkReply *networkReply = network->sendCustomRequest(req, "MKCOL", QByteArray());
    connect(networkReply, &QNetworkReply::sslErrors, this, [=]() {
        qDebug() << "WARNING: The connection to the Nextcloud server is not secure!";
        networkReply->ignoreSslErrors();
    });
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        qDebug() << restReply.readBody();
        return true;
    } else {
        qDebug() << readReply(restReply, "create folder");
    }
    return false;
}

QString Nextcloud::getSharedLink(QString path)
{
    initCall();

    QNetworkRequest req = ncApi.createRequest(QString("shares?path=%1").arg(path));
    QHttpHeaders headers;
    headers.append("OCS-APIRequest", "true");
    headers.append(QHttpHeaders::WellKnownHeader::Accept, "application/json");
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Basic "_s + ncAuth->token());
    req.setHeaders(headers);

    QNetworkReply *networkReply = network->get(req);
    connect(networkReply, &QNetworkReply::sslErrors, this, [=]() {
        qDebug() << "WARNING: The connection to the Nextcloud server is not secure!";
        networkReply->ignoreSslErrors();
    });
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QString sharedLink = "";
    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        qDebug() << restReply.readBody();
        if (auto doc = restReply.readJson(); doc && doc->isObject()) {
            if (doc->object().contains("links")) {
                QJsonArray array = doc->object().value("links").toArray();
                if (array.count() > 0) {
                    QString pathLower = array[0].toObject().value("path_lower").toString();
                    if (path.compare(pathLower, Qt::CaseInsensitive) == 0)
                        sharedLink = array[0].toObject().value("url").toString();
                }
            }
        }
    } else {
        qDebug() << readReply(restReply, "list shared links");
    }

    if (sharedLink.isEmpty()) {
        req = ncApi.createRequest(QString("shares?path=%1").arg(path));
        req.setHeaders(headers);
        networkReply = network->post(req, QByteArray());
        connect(networkReply, &QNetworkReply::sslErrors, this, [=]() {
            qDebug() << "WARNING: The connection to the Nextcloud server is not secure!";
            networkReply->ignoreSslErrors();
        });
        connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
        loop.exec();

        QRestReply restReply(networkReply);
        if (restReply.isSuccess()) {
            qDebug() << restReply.readBody();
            if (auto doc = restReply.readJson(); doc && doc->isObject()) {
                if (doc->object().contains("url"))
                    sharedLink = doc->object().value("url").toString();
            }
        } else {
            qDebug() << readReply(restReply, "create shared link");
        }
    }
    return sharedLink;
}

bool Nextcloud::fileExists(QString path)
{
    initCall();

    QNetworkRequest req = ncApi.createRequest(QString("remote.php/dav/files/%1%2%3")
                                                      .arg(account()->loginName())
                                                      .arg(path.startsWith("/") ? "" : "/")
                                                      .arg(path));
    QHttpHeaders headers;
    headers.append("OCS-APIRequest", "true");
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Basic "_s + ncAuth->token());
    req.setHeaders(headers);

    QByteArray postData = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>"
                          "<d:propfind xmlns:d=\"DAV:\" xmlns:oc=\"http://owncloud.org/ns\" xmlns:nc=\"http://nextcloud.org/ns\">"
                          "  <d:prop>"
                          "    <d:getlastmodified/>"
                          "    <d:getcontentlength/>"
                          "    <d:getcontenttype/>"
                          "    <oc:permissions/>"
                          "    <d:resourcetype/>"
                          "    <d:getetag/>"
                          "  </d:prop>"
                          "</d:propfind>";
    QNetworkReply *networkReply = network->sendCustomRequest(req, "PROPFIND", postData);
    connect(networkReply, &QNetworkReply::sslErrors, this, [=]() {
        qDebug() << "WARNING: The connection to the Nextcloud server is not secure!";
        networkReply->ignoreSslErrors();
    });
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        QDomDocument doc("xml");
        doc.setContent(restReply.readBody());
        QDomNode responseElement = doc.documentElement().firstChild();
        QDomElement hrefElement = responseElement.firstChildElement(hrefTag);

        return !hrefElement.isNull();
    } else {
        qDebug() << readReply(restReply, "get metadata");
    }
    return false;
}

void Nextcloud::loadAccount()
{
    if (!account())
        setAccount(new StorageAccount(StorageServiceProvider::Nextcloud, this));

    if (isAuthenticated() && (settings.value("nextcloud/email", "").toString().isEmpty() || settings.value("nextcloud/name", "").toString().isEmpty())) {
        qDebug() << "Update Nextcloud account";
        initCall();
        qDebug() << "Load Nextcloud account:" << account()->loginName();

        QNetworkRequest req = ncApi.createRequest(QString("ocs/v1.php/cloud/users/%1?format=json").arg(account()->loginName()));
        QHttpHeaders headers;
        headers.append("OCS-APIRequest", "true");
        headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Basic "_s + ncAuth->token());
        req.setHeaders(headers);

        QNetworkReply *networkReply = network->get(req);
        connect(networkReply, &QNetworkReply::sslErrors, this, [=]() {
            qDebug() << "WARNING: The connection to the Nextcloud server is not secure!";
            networkReply->ignoreSslErrors();
        });
        QEventLoop loop;
        connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
        loop.exec();

        QRestReply restReply(networkReply);
        if (restReply.isSuccess()) {
            if (auto doc = restReply.readJson(); doc && doc->isObject()) {
                QJsonObject ocsObj = doc->object().value("ocs").toObject();
                QJsonObject dataObj = ocsObj.value("data").toObject();
                QSettings settings;
                account()->setName(dataObj.value("displayname").toString());
                settings.setValue("nextcloud/name", account()->name());
                account()->setEmail(dataObj.value("email").toString());
                settings.setValue("nextcloud/email", account()->email());
            }
        } else {
            qDebug() << readReply(restReply, "get current account");
        }
    } else {
        account()->setName(settings.value("nextcloud/name", "").toString());
        account()->setEmail(settings.value("nextcloud/email", "").toString());
    }
    account()->setSyncFile(settings.value("nextcloud/syncfile", "").toString());
}

QAbstractItemModel *Nextcloud::searchFile(QString query)
{
    initCall();

    QNetworkRequest req = ncApi.createRequest("remote.php/dav/");
    QHttpHeaders headers;
    headers.append("OCS-APIRequest", "true");
    headers.append(QHttpHeaders::WellKnownHeader::ContentType, "text/xml");
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Basic "_s + ncAuth->token());
    req.setHeaders(headers);

    QByteArray postData = QString("<?xml version=\"1.0\" encoding=\"UTF-8\"?>"
                                  "<d:searchrequest xmlns:d=\"DAV:\" xmlns:oc=\"http://owncloud.org/ns\" xmlns:nc=\"http://nextcloud.org/ns\">"
                                  "  <d:basicsearch>"
                                  "    <d:select>"
                                  "      <d:prop>"
                                  "        <oc:fileid/>"
                                  "        <d:displayname/>"
                                  "        <oc:owner-id/>"
                                  "        <oc:owner-display-name/>"
                                  "        <nc:share-attributes/>"
                                  "        <nc:sharees/>"
                                  "      </d:prop>"
                                  "    </d:select>"
                                  "    <d:from>"
                                  "      <d:scope>"
                                  "        <d:href>/files/%1</d:href>"
                                  "        <d:depth>infinity</d:depth>"
                                  "      </d:scope>"
                                  "    </d:from>"
                                  "    <d:where>"
                                  "      <d:like>"
                                  "        <d:prop>"
                                  "          <d:displayname/>"
                                  "        </d:prop>"
                                  "        <d:literal>%2</d:literal>"
                                  "      </d:like>"
                                  "    </d:where>"
                                  "    <d:orderby/>"
                                  "  </d:basicsearch>"
                                  "</d:searchrequest>")
                                  .arg(account()->loginName())
                                  .arg(query)
                                  .toUtf8();
    QNetworkReply *networkReply = network->sendCustomRequest(req, "SEARCH", postData);
    connect(networkReply, &QNetworkReply::sslErrors, this, [=]() {
        qDebug() << "WARNING: The connection to the Nextcloud server is not secure!";
        networkReply->ignoreSslErrors();
    });
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QHash<int, QByteArray> roles;
    roles[Qt::UserRole + 1] = "id";
    roles[Qt::UserRole + 2] = "name";
    roles[Qt::UserRole + 3] = "path";
    roles[Qt::UserRole + 4] = "sharedby";
    QStandardItemModel *model = new QStandardItemModel(0, 4, this);
    model->setItemRoleNames(roles);

    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        QDomDocument doc("xml");
        doc.setContent(restReply.readBody());
        QDomNodeList responses = doc.documentElement().elementsByTagName(responseTag);
        for (int iResponse = 0; iResponse < responses.count(); iResponse++) {
            QDomNode responseNode = responses.at(iResponse);
            QDomElement hrefElement = responseNode.firstChildElement(hrefTag);
            // save relative path
            qDebug() << hrefElement.text();
            QString path = hrefElement.text().replace(QString("remote.php/dav/files/%1/").arg(account()->loginName()), "");
            // find other properties
            QString id;
            QString name;
            QString sharedBy;
            bool foundSharedBy = false;
            QDomNodeList propstats = doc.documentElement().elementsByTagName(propStatTag);
            for (int iPropstat = 0; iPropstat < propstats.count(); iPropstat++) {
                QDomNode propstatNode = propstats.at(iPropstat);
                QDomElement statusElement = propstatNode.firstChildElement(statusTag);
                qDebug() << propstatNode.toDocument().toString();
                if (statusElement.text().contains("OK")) {
                    // check values of properties found
                    QDomElement propElement = propstatNode.firstChildElement(propTag);
                    QDomNode propChildNode = propElement.firstChild();
                    while (!propChildNode.isNull()) {
                        QDomElement propElement = propChildNode.toElement();
                        if (propElement.tagName().compare(fileIdTag, Qt::CaseInsensitive) == 0)
                            id = propElement.text();
                        if (propElement.tagName().compare(displayNameTag, Qt::CaseInsensitive) == 0)
                            name = propElement.text();
                        if (propElement.tagName().compare(ownerIdTag, Qt::CaseInsensitive) == 0) {
                            qDebug() << ownerIdTag << propElement.text();
                        }
                        if (propElement.tagName().compare(ownerDisplayNameTag, Qt::CaseInsensitive) == 0) {
                            sharedBy = propElement.text();
                            foundSharedBy = true;
                        }
                        if (propElement.tagName().compare(shareesTag, Qt::CaseInsensitive) == 0) {
                            qDebug() << shareesTag << propElement.text();
                        }
                        propChildNode = propChildNode.nextSibling();
                    }
                }
            }
            if (!foundSharedBy && !id.isEmpty())
                sharedBy = getFileOwner(path);

            QStandardItem *item = new QStandardItem();
            item->setData(id, roles.key("id"));
            item->setData(name, roles.key("name"));
            item->setData(path, roles.key("path"));
            item->setData(sharedBy, roles.key("sharedby"));

            model->setRowCount(model->rowCount() + 1);
            model->setItem(model->rowCount() - 1, item);
        }
    } else {
        qDebug() << readReply(restReply, "search file");
    }
    return model;
}

QList<StorageUser> Nextcloud::listFolderMembers(QString path)
{
    initCall();

    QNetworkRequest req = ncApi.createRequest(QString("ocs/v2.php/apps/files_sharing/api/v1/shares?path=%1").arg(path));
    QHttpHeaders headers;
    headers.append("OCS-APIRequest", "true");
    headers.append(QHttpHeaders::WellKnownHeader::Accept, "application/json");
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Basic "_s + ncAuth->token());
    req.setHeaders(headers);

    QNetworkReply *networkReply = network->get(req);
    connect(networkReply, &QNetworkReply::sslErrors, this, [=]() {
        qDebug() << "WARNING: The connection to the Nextcloud server is not secure!";
        networkReply->ignoreSslErrors();
    });
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QList<StorageUser> list;
    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        if (auto doc = restReply.readJson(); doc && doc->isObject()) {
            QJsonObject ocsObj = doc->object().value("ocs").toObject();
            QJsonArray dataArray = ocsObj["data"].toArray();
            for (QJsonValue v : std::as_const(dataArray)) {
                QJsonObject dataObj = v.toObject();
                StorageUser usr;
                usr.displayName = dataObj["share_with_displayname"].toString();
                usr.email = dataObj["share_with_displayname_unique"].toString();
                QString fileOwnerUid = dataObj["uid_file_owner"].toString();
                usr.owner = fileOwnerUid.compare(account()->loginName(), Qt::CaseInsensitive) == 0;
                list.append(usr);
            }
        }
    } else {
        qDebug() << readReply(restReply, "get metadata");
    }
    return list;
}

QString Nextcloud::getFileOwner(QString path)
{
    initCall();

    QNetworkRequest req = ncApi.createRequest(QString("remote.php/dav/files/%1%2%3")
                                                      .arg(account()->loginName())
                                                      .arg(path.startsWith("/") ? "" : "/")
                                                      .arg(path));
    QHttpHeaders headers;
    headers.append("OCS-APIRequest", "true");
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Basic "_s + ncAuth->token());
    req.setHeaders(headers);

    QByteArray postData = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>"
                          "<d:propfind xmlns:d=\"DAV:\" xmlns:oc=\"http://owncloud.org/ns\">"
                          "  <d:prop>"
                          "    <oc:owner-display-name/>"
                          "  </d:prop>"
                          "</d:propfind>";
    QNetworkReply *networkReply = network->sendCustomRequest(req, "PROPFIND", postData);
    connect(networkReply, &QNetworkReply::sslErrors, this, [=]() {
        qDebug() << "WARNING: The connection to the Nextcloud server is not secure!";
        networkReply->ignoreSslErrors();
    });
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QString owner;
    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        QDomDocument doc("xml");
        doc.setContent(restReply.readBody());
        QDomNodeList propstats = doc.elementsByTagName(propStatTag);
        for (int iPropstat = 0; iPropstat < propstats.count(); iPropstat++) {
            QDomNode propstatNode = propstats.at(iPropstat);
            QDomElement statusElement = propstatNode.firstChildElement(statusTag);
            qDebug() << propstatNode.toDocument().toString();
            if (statusElement.text().contains("OK")) {
                // check the values of properties found
                QDomElement propElement = propstatNode.firstChildElement(propTag);
                QDomNode propChildNode = propElement.firstChild();
                while (!propChildNode.isNull()) {
                    QDomElement propElement = propChildNode.toElement();
                    if (propElement.tagName().compare(ownerDisplayNameTag, Qt::CaseInsensitive) == 0) {
                        owner = propElement.firstChild().nodeValue();
                    }
                    propChildNode = propChildNode.nextSibling();
                }
            }
        }
    } else {
        qDebug() << readReply(restReply, "list file members");
    }
    return owner;
}

void Nextcloud::initCall()
{
    m_error = "";

    // check the token expiration
    if (QDateTime::currentDateTime() > tokenExpiration && !ncAuth->refreshToken().isEmpty()) {
        refreshAccessToken();
    }
}

bool Nextcloud::refreshAccessToken()
{
    qDebug() << "refreshAccessToken";

    QNetworkRequest req(ncAuth->tokenUrl());
    req.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");

    // QUrl params;
    QUrlQuery query;
    query.addQueryItem("grant_type", "refresh_token");
    query.addQueryItem("refresh_token", ncAuth->refreshToken());
    query.addQueryItem("client_id", ncAuth->clientIdentifier());
    const QString postData = query.toString(QUrl::FullyEncoded);
    QNetworkReply *networkReply = network->post(req, postData.toUtf8());
    connect(networkReply, &QNetworkReply::sslErrors, this, [=]() {
        qDebug() << "WARNING: The connection to the Nextcloud server is not secure!";
        networkReply->ignoreSslErrors();
    });
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        if (auto doc = restReply.readJson(); doc && doc->isObject()) {
            if (doc->object().contains("access_token")) {
                saveSettings(doc.value());
                qDebug() << "access token updated";

                emit ncAuth->granted();
                return true;
            }
        }
    } else {
        qDebug() << readReply(restReply, "refresh token");
    }
    return false;
}

void Nextcloud::saveSettings(const QJsonDocument doc)
{
    if (!doc.isEmpty()) {
        if (doc.object().contains("access_token")) {
            ncAuth->setToken(doc.object().value("access_token").toString());
            tokenExpiration = QDateTime::currentDateTime().addSecs(doc.object().value("expires_in").toInt() - 10);
            if (doc.object().contains("refresh_token"))
                ncAuth->setRefreshToken(doc.object().value("refresh_token").toString());
        }
    } else {
        if (ncAuth->expirationAt() > QDateTime())
            tokenExpiration = ncAuth->expirationAt();
    }
    settings.setValue("nextcloud/token", ncAuth->token());
    settings.setValue("nextcloud/expirationAt", tokenExpiration);
    settings.setValue("nextcloud/refreshtoken", ncAuth->refreshToken());
}

bool Nextcloud::getAccessToken(QString code)
{
    qDebug() << "getAccessToken";

    QNetworkRequest req(ncAuth->tokenUrl());
    req.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");

    QUrlQuery query;
    query.addQueryItem("code", code);
    query.addQueryItem("grant_type", "authorization_code");
    query.addQueryItem("client_id", ncAuth->clientIdentifier());
    query.addQueryItem("redirect_uri", "theocbase://dropbox_callback");
    const QString postData = query.toString(QUrl::FullyEncoded);

    QNetworkReply *networkReply = network->post(req, postData.toUtf8());

    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        if (auto doc = restReply.readJson(); doc && doc->isObject()) {
            if (doc->object().contains("access_token")) {
                saveSettings(doc.value());
                setToken(ncAuth->token());
                return true;
            }
        }
    } else {
        qDebug() << readReply(restReply, "acquire access token");
    }
    return false;
}

QString Nextcloud::readReply(QRestReply &restReply, QString apiRequest)
{
    QString errorMessage;
    if (restReply.hasError())
        errorMessage = QString("Nextcloud request '%1' failed due to network error [%2]: [%3]")
                               .arg(apiRequest)
                               .arg(restReply.error())
                               .arg(restReply.errorString());
    else {
        QByteArrayView contentType = restReply.networkReply()->headers().value(QHttpHeaders::WellKnownHeader::ContentType);
        int maintenanceMode = restReply.networkReply()->headers().value("x-nextcloud-maintenance-mode").toInt();
        if (restReply.httpStatus() == 503 && maintenanceMode == 1) {
            // If Nextcloud is down for maintenance,
            // it sends a HTTP response with status code 503 and the header x-nextcloud-maintenance-mode: 1
            errorMessage = QString("Nextcloud request '%1' failed [%2]: [%3]")
                                   .arg(apiRequest)
                                   .arg(restReply.httpStatus())
                                   .arg(maintenanceMode == 1 ? "Nextcloud is down for maintenance" : restReply.readText());
        } else {
            QString replyText = restReply.readText();
            if (contentType.contains("html")) {
                QTextDocument *doc = new QTextDocument();
                doc->setHtml(replyText);
                replyText = doc->toPlainText();
            }
            errorMessage = QString("Nextcloud request '%1' failed [%2]: [%3]")
                                   .arg(apiRequest)
                                   .arg(restReply.httpStatus())
                                   .arg(replyText);
        }
    }
    return errorMessage;
}
