#include "dropbox.h"
#include "qrestreply.h"
#include "qtimer.h"
#include <QDesktopServices>

using namespace Qt::StringLiterals;

static constexpr auto apiBaseUrl = "https://api.dropboxapi.com/2"_L1;
static constexpr auto contentApiBaseUrl = "https://content.dropboxapi.com/2"_L1;
static constexpr auto authorizationUrl = "https://www.dropbox.com/oauth2/authorize"_L1;
static constexpr auto accessTokenUrl = "https://api.dropboxapi.com/oauth2/token"_L1;

Dropbox::Dropbox(const QString &clientId, QObject *parent)
    : StorageService(StorageServiceProvider::Dropbox, parent)
{
    dropboxAuth = new QOAuth2AuthorizationCodeFlow(this);
    connect(dropboxAuth, &QOAuth2AuthorizationCodeFlow::tokenChanged, this, [=](const QString token) {
        setIsAuthenticated(!token.isEmpty());
    });
    QNetworkAccessManager *qnam = new QNetworkAccessManager(this);
    network = new QRestAccessManager(qnam, qnam);

    dropboxApi.setBaseUrl(QUrl(apiBaseUrl));
    dropboxContentApi.setBaseUrl(QUrl(contentApiBaseUrl));

    dropboxAuth->setAuthorizationUrl(QUrl(authorizationUrl));
    dropboxAuth->setRequestedScopeTokens({ "account_info.read", "files.metadata.write", "files.metadata.read", "files.content.write", "files.content.read", "sharing.write", "sharing.read" });
    dropboxAuth->setTokenUrl(QUrl(accessTokenUrl));
    dropboxAuth->setModifyParametersFunction([&](QAbstractOAuth::Stage stage, QMultiMap<QString, QVariant> *params) {
        if (stage == QAbstractOAuth::Stage::RequestingAuthorization) {
            qDebug() << "Dropbox-API modify parameters function (RequestingAuthorization)";
            params->insert("token_access_type", "offline"); // this is for code flow with offline token and not PKCE code flow (?)
        } else if (stage == QAbstractOAuth::Stage::RefreshingAccessToken) {
            qDebug() << "Dropbox-API modify parameters function (RefreshingAccessToken)";
            params->remove("redirect_uri");
        } else if (stage == QAbstractOAuth::Stage::RequestingAccessToken) {
            qDebug() << "Dropbox-API modify parameters function (RequestingAccessToken)";
        }
    });

    dropboxAuth->setClientIdentifier(clientId);

    // load previous login settings
    QString token = settings.value("dropbox/token", "").toString();
    if (!token.isEmpty()) {
        QString email = settings.value("dropbox/email", "").toString();
        QString name = settings.value("dropbox/name", "").toString();
        setAccount(new StorageAccount(StorageServiceProvider::Dropbox, name, email, this)); // set login name
        account()->setSyncFile(settings.value("dropbox/syncfile", "").toString());
    }
    dropboxAuth->setToken(token);

    QString refreshToken = settings.value("dropbox/refreshtoken").toString();
    dropboxAuth->setRefreshToken(refreshToken);
    tokenExpiration = settings.value("dropbox/expirationAt").toDateTime();

    connect(dropboxAuth, &QAbstractOAuth::granted, this, [=]() {
        qDebug() << "DropboxAPI granted";
        dropboxApi.setBearerToken(dropboxAuth->token().toLatin1());
        dropboxContentApi.setBearerToken(dropboxAuth->token().toLatin1());
        saveSettings(QJsonDocument());
#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
        uriSchemeReplyHandler->close();
#endif
    });
    connect(dropboxAuth, &QOAuth2AuthorizationCodeFlow::authorizeWithBrowser, this, &QDesktopServices::openUrl);
    // provide a signal e.g. to close the Dropbox panel
    connect(dropboxAuth, &QOAuth2AuthorizationCodeFlow::authorizeWithBrowser, this, &Dropbox::authorizeWithBrowserStarted);
    connect(dropboxAuth, &QOAuth2AuthorizationCodeFlow::statusChanged, this, &Dropbox::statusChanged);
    // OAuth Error Handling
    connect(dropboxAuth, &QOAuth2AuthorizationCodeFlow::serverReportedErrorOccurred, [](const QString &error, const QString &errorDescription, const QUrl &uri) {
        qDebug() << "OAuth Error:" << error;
        qDebug() << "Error Description:" << errorDescription;
        qDebug() << "URI:" << uri;
    });

#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
    uriSchemeReplyHandler = new QOAuthUriSchemeReplyHandler(QUrl { "theocbase://dropbox_callback"_L1 }, this);
    dropboxAuth->setReplyHandler(uriSchemeReplyHandler);
    // // Initiate the authorization
    // if (uriSchemeReplyHandler->listen()) {
    //     dropboxAuth->grant();
    // }
#else
    auto replyHandler = new QOAuthHttpServerReplyHandler(55738, this);
    replyHandler->setCallbackPath("/dropbox");
    dropboxAuth->setReplyHandler(replyHandler);
#endif
}

void Dropbox::authenticate()
{
    qDebug() << "authenticate";
    if (!dropboxAuth->token().isEmpty()) {
        qDebug() << "Already authenticated!";
        return;
    }

// Initiate the authorization
#if defined(Q_OS_ANDROID) || defined(Q_OS_IOS)
    if (uriSchemeReplyHandler->listen()) {
        dropboxAuth->grant();
    }
#else
    dropboxAuth->grant();
#endif
}

void Dropbox::revoke()
{
    qDebug() << "revoke";
    initCall();

    QNetworkRequest req = dropboxApi.createRequest("auth/token/revoke");
    QHttpHeaders headers;
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Bearer "_s + dropboxAuth->token());
    req.setHeaders(headers);

    QByteArray postData = "null";
    QNetworkReply *networkReply = network->post(req, postData);
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        qDebug() << "revoke finished successfully";
    } else {
        qDebug() << readReply(restReply, "revoke token");
    }

    // clear settings
    settings.setValue("dropbox/refreshtoken", "");
    settings.setValue("dropbox/name", "");
    settings.setValue("dropbox/email", "");
    settings.setValue("dropbox/syncfile", "");
    account()->reset();
    // clear token and trigger authorized changed signal
    this->setToken("");
}

void Dropbox::setToken(const QString token)
{
    qDebug() << "setToken";
    settings.setValue("dropbox/token", token);
    dropboxAuth->setToken(token);
    // QAbstractOAuth::Status newstatus = token.isEmpty() ? QAbstractOAuth::Status::NotAuthenticated : QAbstractOAuth::Status::Granted;
    // if (dropboxAuth->status() != newstatus)
    //     emit dropboxAuth->statusChanged(newstatus);
}

QString Dropbox::errorString()
{
    return m_error;
}

QDateTime Dropbox::upload(QString localFileName, QString relativeCloudFileName)
{
    QFile file(localFileName);
    if (!file.open(QIODevice::ReadOnly))
        return QDateTime();
    QByteArray postData = file.readAll();
    file.close();

    return upload(postData, relativeCloudFileName);
}

QDateTime Dropbox::upload(QByteArray data, QString relativeCloudFileName)
{
    initCall();

    QNetworkRequest req = dropboxContentApi.createRequest("files/upload");
    QHttpHeaders headers;
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Bearer "_s + dropboxAuth->token());
    // relativeCloudFileName will need URL encoded here, I think
    headers.append("Dropbox-API-Arg", QString("{\"path\":\"%1\",\"mode\":\"overwrite\",\"autorename\": true, \"mute\": true }").arg(relativeCloudFileName).toUtf8());
    headers.append(QHttpHeaders::WellKnownHeader::ContentType, "application/octet-stream");
    req.setHeaders(headers);

    QNetworkReply *networkReply = network->post(req, data);
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QDateTime dt;
    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        if (auto doc = restReply.readJson(); doc && doc->isObject()) {
            if (m_error.isEmpty()) {
                qDebug() << doc;
                qDebug() << doc->object().value("server_modified").toString();
                dt = QDateTime::fromString(doc->object().value("server_modified").toString(), Qt::ISODate);
            }
        }
    } else {
        qDebug() << readReply(restReply, "upload file");
    }
    return dt;
}

void Dropbox::download(QString relativeCloudFileName, QString localFileName, QDateTime &modifiedDate)
{
    QByteArray data;
    download(relativeCloudFileName, data, modifiedDate);

    QFile file(localFileName);
    if (file.open(QIODevice::WriteOnly)) {
        file.write(data);
        file.close();
    }
}

void Dropbox::download(QString relativeCloudFileName, QByteArray &content, QDateTime &modifiedDate)
{
    initCall();

    QNetworkRequest req = dropboxContentApi.createRequest("files/download");
    QHttpHeaders headers;
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Bearer "_s + dropboxAuth->token());
    // relativeCloudFileName will need URL encoded here, I think
    QString temp = QString("{\"path\":\"%1\"}").arg(relativeCloudFileName);
    headers.append("Dropbox-API-Arg", temp.toUtf8());
    headers.append(QHttpHeaders::WellKnownHeader::ContentType, "application/octet-stream");
    req.setHeaders(headers);

    QNetworkReply *networkReply = network->post(req, QByteArray());
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QDateTime dt;
    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        // read file content
        content = restReply.readBody();
        // read file meta data
        QJsonDocument doc = QJsonDocument::fromJson(restReply.networkReply()->rawHeader("Dropbox-API-Result"));
        modifiedDate = QDateTime::fromString(doc.object().value("server_modified").toString(), Qt::ISODate);
    } else {
        qDebug() << readReply(restReply, "download file");
    }
}

QDateTime Dropbox::getModifiedDate(QString cloudFileName)
{
    initCall();

    QNetworkRequest req = dropboxApi.createRequest("files/get_metadata");
    QHttpHeaders headers;
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Bearer "_s + dropboxAuth->token());
    req.setHeaders(headers);

    QString temp = QString("{\"path\":\"%1\"}").arg(cloudFileName);
    QJsonDocument postData = QJsonDocument::fromJson(temp.toUtf8());

    QNetworkReply *networkReply = network->post(req, postData);
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        if (auto doc = restReply.readJson(); doc && doc->isObject()) {
            QDateTime dt = QDateTime::fromString(doc->object().value("server_modified").toString(), Qt::ISODate);
            return dt;
        }
    } else {
        qDebug() << readReply(restReply, "get file metadata");
    }
    return QDateTime();
}

bool Dropbox::deleteFile(QString cloudFileName)
{
    initCall();

    QNetworkRequest req = dropboxApi.createRequest("files/delete_v2");
    QHttpHeaders headers;
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Bearer "_s + dropboxAuth->token());
    req.setHeaders(headers);

    QString temp = QString("{\"path\":\"%1\"}").arg(cloudFileName);
    QJsonDocument postData = QJsonDocument::fromJson(temp.toUtf8());
    QNetworkReply *networkReply = network->post(req, postData);
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        if (auto doc = restReply.readJson(); doc && doc->isObject()) {
            return doc->object().contains("metadata");
        }
    } else {
        qDebug() << readReply(restReply, "delete file");
    }
    return false;
}

bool Dropbox::createFolder(QString path)
{
    initCall();

    QNetworkRequest req = dropboxApi.createRequest("files/create_folder_v2");
    QHttpHeaders headers;
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Bearer "_s + dropboxAuth->token());
    req.setHeaders(headers);

    QString temp = QString("{\"path\":\"%1\"}").arg(path);
    QJsonDocument postData = QJsonDocument::fromJson(temp.toUtf8());
    QNetworkReply *networkReply = network->post(req, postData);
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        if (auto doc = restReply.readJson(); doc && doc->isObject()) {
            return doc->object().contains("metadata");
        }
    } else {
        qDebug() << readReply(restReply, "create folder");
    }
    return false;
}

QString Dropbox::getSharedLink(QString path)
{
    initCall();

    QNetworkRequest req = dropboxApi.createRequest("sharing/list_shared_links");
    QHttpHeaders headers;
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Bearer "_s + dropboxAuth->token());
    req.setHeaders(headers);

    QString temp = QString("{\"path\":\"%1\"}").arg(path);
    QJsonDocument postData = QJsonDocument::fromJson(temp.toUtf8());
    QNetworkReply *networkReply = network->post(req, postData);
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QString sharedLink = "";
    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        if (auto doc = restReply.readJson(); doc && doc->isObject()) {
            if (doc->object().contains("links")) {
                QJsonArray array = doc->object().value("links").toArray();
                if (array.count() > 0) {
                    QString pathLower = array[0].toObject().value("path_lower").toString();
                    if (path.compare(pathLower, Qt::CaseInsensitive) == 0)
                        sharedLink = array[0].toObject().value("url").toString();
                }
            }
        }
    } else {
        qDebug() << readReply(restReply, "list shared links");
    }

    if (sharedLink.isEmpty()) {
        req = dropboxApi.createRequest("sharing/create_shared_link_with_settings");
        req.setHeaders(headers);
        networkReply = network->post(req, postData);
        connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
        loop.exec();

        QRestReply restReply(networkReply);
        if (restReply.isSuccess()) {
            if (auto doc = restReply.readJson(); doc && doc->isObject()) {
                if (doc->object().contains("url"))
                    sharedLink = doc->object().value("url").toString();
            }
        } else {
            qDebug() << readReply(restReply, "create shared link");
        }
    }
    return sharedLink;
}

bool Dropbox::fileExists(QString path)
{
    initCall();

    QNetworkRequest req = dropboxApi.createRequest("files/get_metadata");
    QHttpHeaders headers;
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Bearer "_s + dropboxAuth->token());
    req.setHeaders(headers);

    QString temp = QString("{\"path\":\"%1\"}").arg(path);
    QJsonDocument postData = QJsonDocument::fromJson(temp.toUtf8());
    QNetworkReply *networkReply = network->post(req, postData);
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        if (auto doc = restReply.readJson(); doc && doc->isObject()) {
            return doc->object().contains("name");
        }
    } else {
        qDebug() << readReply(restReply, "get metadata");
    }
    return false;
}

void Dropbox::loadAccount()
{
    if (!account())
        setAccount(new StorageAccount(StorageServiceProvider::Dropbox, this));

    if (isAuthenticated() && (settings.value("dropbox/email", "").toString().isEmpty() || settings.value("dropbox/name", "").toString().isEmpty())) {
        qDebug() << "Update Dropbox account";
        initCall();
        qDebug() << "Load current account from Dropbox";

        QNetworkRequest req = dropboxApi.createRequest("users/get_current_account");
        QHttpHeaders headers;
        headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Bearer "_s + dropboxAuth->token());
        req.setHeaders(headers);

        QByteArray postData = "null";
        QNetworkReply *networkReply = network->post(req, postData);
        QEventLoop loop;
        connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
        loop.exec();

        QRestReply restReply(networkReply);
        if (restReply.isSuccess()) {
            if (auto doc = restReply.readJson(); doc && doc->isObject()) {
                QSettings settings;
                account()->setName(doc->object().value("name").toObject().value("display_name").toString());
                settings.setValue("dropbox/name", account()->name());
                account()->setEmail(doc->object().value("email").toString());
                settings.setValue("dropbox/email", account()->email());
            }
        } else {
            qDebug() << readReply(restReply, "get current account");
        }
    } else {
        account()->setName(settings.value("dropbox/name", "").toString());
        account()->setEmail(settings.value("dropbox/email", "").toString());
    }
    account()->setSyncFile(settings.value("dropbox/syncfile", "").toString());
}

QAbstractItemModel *Dropbox::searchFile(QString query)
{
    initCall();

    QNetworkRequest req = dropboxApi.createRequest("files/search_v2");
    QHttpHeaders headers;
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Bearer "_s + dropboxAuth->token());
    req.setHeaders(headers);

    QString temp = QString("{\"query\":\"%1\", \"options\":{ \"filename_only\":true }}").arg(query);
    QJsonDocument postData = QJsonDocument::fromJson(temp.toUtf8());
    QNetworkReply *networkReply = network->post(req, postData);
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QHash<int, QByteArray> roles;
    roles[Qt::UserRole + 1] = "id";
    roles[Qt::UserRole + 2] = "name";
    roles[Qt::UserRole + 3] = "path";
    roles[Qt::UserRole + 4] = "sharedby";
    QStandardItemModel *model = new QStandardItemModel(0, 4, this);
    model->setItemRoleNames(roles);

    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        if (auto doc = restReply.readJson(); doc && doc->isObject()) {
            QJsonArray array = doc->object()["matches"].toArray();
            for (const QJsonValue &v : std::as_const(array)) {
                QJsonObject metaDataRoot = v.toObject().value("metadata").toObject();
                QJsonObject metadata = metaDataRoot.value("metadata").toObject();
                if (metadata.contains("path_display")) {
                    QString id = metadata.value("id").toString();
                    QString name = metadata.value("name").toString();
                    QString path = metadata.value("path_display").toString();
                    QString sharedBy = metadata.contains("sharing_info") ? getFileOwner(id) : "";

                    QStandardItem *item = new QStandardItem();
                    item->setData(id, roles.key("id"));
                    item->setData(name, roles.key("name"));
                    item->setData(path, roles.key("path"));
                    item->setData(sharedBy, roles.key("sharedby"));

                    model->setRowCount(model->rowCount() + 1);
                    model->setItem(model->rowCount() - 1, item);
                }
            }
        }
    } else {
        qDebug() << readReply(restReply, "search file");
    }
    return model;
}

QList<StorageUser> Dropbox::listFolderMembers(QString path)
{
    initCall();

    // 1. get shared folder id
    QNetworkRequest req = dropboxApi.createRequest("files/get_metadata");
    QHttpHeaders headers;
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Bearer "_s + dropboxAuth->token());
    req.setHeaders(headers);

    QString args = QString("{\"path\":\"%1\"}").arg(path);
    QJsonDocument postData = QJsonDocument::fromJson(args.toUtf8());
    QNetworkReply *networkReply = network->post(req, postData);
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QString sharedFolderId;
    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        if (auto doc = restReply.readJson(); doc && doc->isObject()) {
            if (doc->object().contains("sharing_info")) {
                QJsonObject sharingObj = doc->object()["sharing_info"].toObject();
                if (sharingObj.contains("shared_folder_id"))
                    sharedFolderId = sharingObj["shared_folder_id"].toString();
                else if (sharingObj.contains("parent_shared_folder_id"))
                    sharedFolderId = sharingObj["parent_shared_folder_id"].toString();
            }
        }
    } else {
        qDebug() << readReply(restReply, "get metadata");
    }

    // 2. get list of folder members
    QList<StorageUser> list;
    if (!sharedFolderId.isEmpty()) {
        args = QString("{\"shared_folder_id\":\"%1\"}").arg(sharedFolderId);
        postData = QJsonDocument::fromJson(args.toUtf8());

        req = dropboxApi.createRequest("sharing/list_folder_members");
        networkReply = network->post(req, postData);

        connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
        loop.exec();

        restReply = QRestReply(networkReply);

        if (restReply.isSuccess()) {
            if (auto doc = restReply.readJson(); doc && doc->isObject()) {
                if (doc->object().contains("users")) {
                    QJsonArray usersArray = doc->object()["users"].toArray();
                    for (QJsonValue v : std::as_const(usersArray)) {
                        QString accessType = v.toObject()["access_type"].toObject()[".tag"].toString();
                        QJsonObject userObj = v.toObject().value("user").toObject();
                        qDebug() << "Dropbox user:" << userObj["display_name"].toString()
                                 << userObj["email"].toString() << accessType;
                        StorageUser usr;
                        usr.displayName = userObj["display_name"].toString();
                        usr.email = userObj["email"].toString();
                        usr.owner = (accessType == "owner");
                        list.append(usr);
                    }
                }
            }
        } else {
            qDebug() << readReply(restReply, "list folder members");
        }
    }
    return list;
}

QString Dropbox::getFileOwner(QString id)
{
    initCall();

    QNetworkRequest req = dropboxApi.createRequest("sharing/list_file_members");
    QHttpHeaders headers;
    headers.append(QHttpHeaders::WellKnownHeader::Authorization, u"Bearer "_s + dropboxAuth->token());
    req.setHeaders(headers);

    QString temp = QString("{\"file\":\"%1\"}").arg(id);
    QJsonDocument postData = QJsonDocument::fromJson(temp.toUtf8());
    QNetworkReply *networkReply = network->post(req, postData);
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QString owner = "";
    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        if (auto doc = restReply.readJson(); doc && doc->isObject()) {
            QJsonArray array = doc->object()["users"].toArray();
            for (const QJsonValue &v : std::as_const(array)) {
                QJsonObject accesstype = v.toObject().value("access_type").toObject();
                QJsonObject user = v.toObject().value("user").toObject();

                if (accesstype.value(".tag").toString() == "owner")
                    owner = user.value("display_name").toString();
            }
        }
    } else {
        qDebug() << readReply(restReply, "list file members");
    }
    return owner;
}

void Dropbox::initCall()
{
    m_error = "";

    // check the token expiration
    if (QDateTime::currentDateTime() > tokenExpiration && !dropboxAuth->refreshToken().isEmpty()) {
        refreshAccessToken();
    }
}

bool Dropbox::refreshAccessToken()
{
    qDebug() << "refreshAccessToken";

    QNetworkRequest req(dropboxAuth->tokenUrl());
    req.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");

    // QUrl params;
    QUrlQuery query;
    query.addQueryItem("grant_type", "refresh_token");
    query.addQueryItem("refresh_token", dropboxAuth->refreshToken());
    query.addQueryItem("client_id", dropboxAuth->clientIdentifier());
    const QString postData = query.toString(QUrl::FullyEncoded);
    QNetworkReply *networkReply = network->post(req, postData.toUtf8());
    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        if (auto doc = restReply.readJson(); doc && doc->isObject()) {
            if (doc->object().contains("access_token")) {
                saveSettings(doc.value());
                qDebug() << "access token updated";

                emit dropboxAuth->granted();
                return true;
            }
        }
    } else {
        qDebug() << readReply(restReply, "refresh token");
    }
    return false;
}

void Dropbox::saveSettings(const QJsonDocument doc)
{
    if (!doc.isEmpty()) {
        if (doc.object().contains("access_token")) {
            dropboxAuth->setToken(doc.object().value("access_token").toString());
            tokenExpiration = QDateTime::currentDateTime().addSecs(doc.object().value("expires_in").toInt() - 10);
            if (doc.object().contains("refresh_token"))
                dropboxAuth->setRefreshToken(doc.object().value("refresh_token").toString());
        }
    } else {
        if (dropboxAuth->expirationAt() > QDateTime())
            tokenExpiration = dropboxAuth->expirationAt();
    }
    settings.setValue("dropbox/token", dropboxAuth->token());
    settings.setValue("dropbox/expirationAt", tokenExpiration);
    settings.setValue("dropbox/refreshtoken", dropboxAuth->refreshToken());
}

bool Dropbox::getAccessToken(QString code)
{
    qDebug() << "getAccessToken";

    QNetworkRequest req(dropboxAuth->tokenUrl());
    req.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");

    QUrlQuery query;
    query.addQueryItem("code", code);
    query.addQueryItem("grant_type", "authorization_code");
    query.addQueryItem("client_id", dropboxAuth->clientIdentifier());
    query.addQueryItem("redirect_uri", "theocbase://dropbox_callback");
    const QString postData = query.toString(QUrl::FullyEncoded);

    QNetworkReply *networkReply = network->post(req, postData.toUtf8());

    QEventLoop loop;
    connect(networkReply, SIGNAL(finished()), &loop, SLOT(quit()));
    loop.exec();

    QRestReply restReply(networkReply);
    if (restReply.isSuccess()) {
        if (auto doc = restReply.readJson(); doc && doc->isObject()) {
            if (doc->object().contains("access_token")) {
                saveSettings(doc.value());
                setToken(dropboxAuth->token());
                return true;
            }
        }
    } else {
        qDebug() << readReply(restReply, "acquire access token");
    }
    return false;
}

QString Dropbox::readReply(QRestReply &restReply, QString apiRequest)
{
    QString errorMessage;
    if (restReply.hasError())
        errorMessage = QString("Dropbox request '%1' failed due to network error [%2]: [%3]")
                               .arg(apiRequest)
                               .arg(restReply.error())
                               .arg(restReply.errorString());
    else {
        switch (restReply.httpStatus()) {
        case 401: {
            if (auto doc = restReply.readJson(); doc && doc->isObject()) {
                // trigger refreshing the token if expired
                QJsonObject errorObj = doc->object().value("error").toObject();
                if (errorObj.value(".tag").toString() == "expired_access_token" && !dropboxAuth->refreshToken().isEmpty())
                    refreshAccessToken();
                else
                    errorMessage = QString("Dropbox request '%1' failed [%2]: [%3]")
                                           .arg(apiRequest)
                                           .arg(restReply.httpStatus())
                                           .arg(doc->toJson());
            }
            break;
        }
        default:
            errorMessage = QString("Dropbox request '%1' failed [%2]: [%3]")
                                   .arg(apiRequest)
                                   .arg(restReply.httpStatus())
                                   .arg(restReply.readText());
            break;
        }
    }
    return errorMessage;
}
