#ifndef CLOUD_CONTROLLER_H
#define CLOUD_CONTROLLER_H

#include <QObject>
#include <QDebug>
#include <QString>
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QNetworkAccessManager>
#include <QCryptographicHash>
#include <QEventLoop>
#include <QJsonDocument>
#include <QJsonArray>
#include <QJsonObject>
#include <QObjectBindableProperty>
#include <QSettings>
#include <QTimer>
#include <QQmlEngine>
#include "../sync_cloud.h"
#include "../sql_class.h"
#include "../constants.h"
#include "nextcloud.h"
#include "../accesscontrol.h"
#include "../cpersons.h"

class cloud_controller : public QObject
{
    Q_OBJECT

    Q_PROPERTY(QDateTime lastSyncTime READ lastSyncTime WRITE setLastSyncTime BINDABLE bindableLastSyncTime)
    Q_PROPERTY(QString backgroundDebug READ debugBackground WRITE setDebugBackground)
    Q_PROPERTY(SyncState syncState READ syncState NOTIFY stateChanged)
    Q_PROPERTY(StorageService *storageService READ storageService BINDABLE bindableStorageService)
    Q_PROPERTY(bool isLoggedIn READ isLoggedIn BINDABLE bindableIsLoggedIn)

public:
    enum SyncState {
        Synchronized,
        Upload,
        Download,
        Both
    };
    Q_ENUM(SyncState)

    cloud_controller(QObject *parent = nullptr, StorageService *service = nullptr);

    Q_INVOKABLE void login();
    Q_INVOKABLE void logout(int clearDB = 1);
    Q_INVOKABLE bool checkCloudUpdates();

    Q_INVOKABLE void synchronize(bool ignoreUser = false);
    Q_INVOKABLE void continueSynchronize(bool keepLocalChanges);
    Q_INVOKABLE void uploadAccessControl();
    Q_INVOKABLE void initAccessControl();
    Q_INVOKABLE void loadAccessControl();
    Q_INVOKABLE void runTest();
    Q_INVOKABLE bool canResetCloudData();
    Q_INVOKABLE void resetCloudData();
    Q_INVOKABLE void clearDatabase();

    QPointer<StorageService> storageService() const { return b_storageService; }
    QBindable<QPointer<StorageService>> bindableStorageService() { return &b_storageService; }

    bool isLoggedIn() { return b_isLoggedIn.value(); }
    void setIsLoggedIn(bool newValue) { b_isLoggedIn = newValue; }
    QBindable<bool> bindableIsLoggedIn() { return &b_isLoggedIn; }

    QDateTime lastSyncTime() { return b_lastSyncTime.value(); }
    void setLastSyncTime(QDateTime newValue) { b_lastSyncTime = newValue; }
    QBindable<bool> bindableLastSyncTime() { return &b_lastSyncTime; }

    SyncState syncState();

    QString debugBackground();
    void setDebugBackground(QString value);

public slots:
    void databaseChanged(const QString tablename);
    void setStorageService(QString newValue, QString baseUrl = "");

signals:
    void authorizeWithBrowserStarted();
    void storageServiceChanged();
    void isLoggedInChanged();
    void syncStarted();
    void syncConflict(int values);
    void syncProgressed(int value, int max);
    void syncFinished();
    void error(QString message);
    void lastSyncTimeChanged();
    void differentLastDbUser();
    void stateChanged(cloud_controller::SyncState state);
    void cloudResetStarted();
    void cloudResetFound();
    void cloudResetFinished();

private:
    const QString defaultOnlineStorageDir = "/Apps/TheocBase/";
    sync_cloud s;
    SyncState mSyncState;
    QAbstractOAuth::Status oauthStatus;

    void setStorageService(QPointer<StorageService> newValue);
    Q_OBJECT_BINDABLE_PROPERTY(cloud_controller, QDateTime, b_lastSyncTime, &cloud_controller::lastSyncTimeChanged);
    Q_OBJECT_BINDABLE_PROPERTY_WITH_ARGS(cloud_controller, QPointer<StorageService>, b_storageService, nullptr, &cloud_controller::storageServiceChanged);
    Q_OBJECT_BINDABLE_PROPERTY(cloud_controller, bool, b_isLoggedIn, &cloud_controller::isLoggedInChanged);

    QMetaObject::Connection authorizeWithBrowserStartedConnection;
    QPropertyNotifier isLoggedInNotifier;
    QPropertyNotifier syncFileNotifier;
    QProperty<QString> b_syncFile;

    void syncTimeCheck();
    void initStorageService();
    void syncCloudConflict(int duplicateValues);
    void syncCloudResetFound();
    const QString cachedAccessControlFile();
    const QString cloudAccessControlFile();
};

#endif // CLOUD_CONTROLLER_H
