#include "cloud_controller.h"
#include "dropbox.h"

cloud_controller::cloud_controller(QObject *parent, StorageService *service)
    : QObject(parent), b_storageService(service)
{
    qmlRegisterType<cloud_controller>("net.theocbase", 1, 0, "Cloud");
    qmlRegisterType<StorageService>("net.theocbase", 1, 0, "StorageService");
    qmlRegisterType<StorageAccount>("net.theocbase", 1, 0, "StorageAccount");

    // create notifier for changes and take into consideration
    // that both the storage service and the account may change
    b_isLoggedIn.setBinding([&]() {
        return storageService() && storageService()->isAuthenticated();
    });
    isLoggedInNotifier = bindableIsLoggedIn().addNotifier([&]() {
        if (isLoggedIn()) {
            initStorageService();
        }
    });
    b_syncFile.setBinding([&]() {
        return storageService() && storageService()->account()
                ? storageService()->account()->syncFile()
                : "";
    });
    syncFileNotifier = b_syncFile.addNotifier([&]() {
        QFile localAccessFile(cachedAccessControlFile());
        if (localAccessFile.exists())
            localAccessFile.remove();
        initAccessControl();
    });

    QSettings settings;
    QString storageServiceProviderName = settings.value("data_storage/online_storage_service", "Dropbox").toString();
    QString storageServiceBaseUrl;
    if (storageServiceProviderName.compare("Nextcloud", Qt::CaseInsensitive) == 0) {
        storageServiceBaseUrl = settings.value("nextcloud/server", "").toString();
    }
    setStorageService(storageServiceProviderName, storageServiceBaseUrl);

    sql_class *sql = &Singleton<sql_class>::Instance();
    connect(&s, &sync_cloud::syncConflict, this, &cloud_controller::syncCloudConflict);
    connect(&s, &sync_cloud::progressed, this, [=](int value) { emit syncProgressed(value, 100); });
    connect(&s, &sync_cloud::ready, this, [=]() {
        sql->saveSetting("local_changes", "false");
        mSyncState = SyncState::Synchronized;
        emit syncFinished();
        syncTimeCheck();
        emit stateChanged(mSyncState);
    });
    connect(&s, &sync_cloud::error, this, &cloud_controller::error);
    connect(&s, &sync_cloud::differentLastDbUser, this, &cloud_controller::differentLastDbUser);
    connect(&s, &sync_cloud::cloudResetFound, this, &cloud_controller::syncCloudResetFound);
    connect(&s, &sync_cloud::cloudResetReady, this, &cloud_controller::cloudResetFinished);

    mSyncState = QVariant(sql->getSetting("local_changes", "false")).toBool() ? SyncState::Upload : SyncState::Synchronized;
    connect(sql, &sql_class::dbChanged, this, &cloud_controller::databaseChanged, Qt::QueuedConnection);

    syncTimeCheck();
}

void cloud_controller::login()
{
    if (!QSslSocket::supportsSsl()) {
        emit error("This device doesn't support SSL.\n"
                   "Make sure that OpenSSL is installed.");
        return;
    }
    storageService()->authenticate();
}

void cloud_controller::logout(int clearDB)
{
    // check current access control permissions before logging out of the storage service
    AccessControl *ac = &Singleton<AccessControl>::Instance();
    bool isStorageServiceOwner = ac->user() && ac->user()->hasPermission(Permission::Rule::CanEditPermissions);
    storageService()->revoke();

    // clear database
    sql_class *sql = &Singleton<sql_class>::Instance();
    sql->blockSignals(true);

    switch (clearDB) {
    case 0:
        // don't clear database
        sql->saveSetting("last_dbsync_id", "0");
        sql->saveSetting("last_dbsync_time", "-1");
        break;
    case 1:
        clearDatabase();
        sql->saveSetting("last_db_user", "");
        break;
    case 2:
        if (!isStorageServiceOwner) {
            // clear database, if not owner of the cloud storage
            clearDatabase();
            sql->saveSetting("last_db_user", "");
        } else {
            // don't clear database, if owner of the cloud storage
            sql->saveSetting("last_dbsync_id", "0");
            sql->saveSetting("last_dbsync_time", "-1");
        }
        break;
    }
    // reset access control
    QFile localAccessControlFile(cachedAccessControlFile());
    if (localAccessControlFile.exists())
        localAccessControlFile.remove();
    QSettings settings;
    settings.setValue("local_access_control", -1);
    initAccessControl();

    sql->blockSignals(false);
    setLastSyncTime(QDateTime());
}

bool cloud_controller::checkCloudUpdates()
{
    bool updates = s.cloudUpdateAvailable();
    if (updates) {
        mSyncState = (mSyncState == SyncState::Upload) ? SyncState::Both : SyncState::Download;
        emit stateChanged(mSyncState);
    }
    qDebug() << "updates checked" << updates;
    return updates;
}

cloud_controller::SyncState cloud_controller::syncState()
{
    return mSyncState;
}

void cloud_controller::synchronize(bool ignoreUser)
{
    emit syncStarted();
    // load access control file
    QSettings settings;
    int localModified = settings.value("local_access_file", -1).toInt();
    QString acFile = cloudAccessControlFile();
    QDateTime dt = storageService()->getModifiedDate(acFile);
    if (dt.toMSecsSinceEpoch() > localModified)
        loadAccessControl();
    // sync database
    s.sync(ignoreUser);
}

void cloud_controller::continueSynchronize(bool keepLocalChanges)
{
    s.continueSync(keepLocalChanges);
}

void cloud_controller::uploadAccessControl()
{
    AccessControl *ac = &Singleton<AccessControl>::Instance();
    QSettings settings;

    QString dbFile = cloudAccessControlFile();
    QJsonObject jsObj;
    ac->write(jsObj);
    QJsonDocument jsDoc(jsObj);
    QDateTime dt = storageService()->upload(jsDoc.toJson(QJsonDocument::Compact), dbFile);
    qDebug() << "access control file uploaed" << dt;
    if (dt.isValid())
        settings.setValue("local_access_control", dt.toSecsSinceEpoch());

    // save cached file
    QFile cacheFile(cachedAccessControlFile());
    if (cacheFile.open(QFile::WriteOnly))
        cacheFile.write(jsDoc.toJson(QJsonDocument::Compact));
}

void cloud_controller::initAccessControl()
{
    AccessControl *ac = &Singleton<AccessControl>::Instance();
    // get publisher and admin roles
    const Role *publisherRole = nullptr;
    const Role *adminRole = nullptr;
    for (const Role &role : ac->roles()) {
        if (role.id() == Permission::RoleId::Publisher)
            publisherRole = &role;
        if (role.id() == Permission::RoleId::Administrator)
            adminRole = &role;
    }
    if (isLoggedIn()) {
        // Logged in to a storage service
        // get storage account info
        StorageAccount *dbUser = storageService()->account();

        QFile localFile(cachedAccessControlFile());
        qDebug() << "cache file" << localFile.fileName();
        bool valid = false;
        if (localFile.exists()) {
            qDebug() << "found cached file";
            // use saved local file
            if (localFile.open(QFile::ReadOnly)) {
                QJsonDocument doc = QJsonDocument::fromJson(localFile.readAll());
                ac->load(doc.object(), valid);
                localFile.close();
            }
            if (valid) {
                ac->setUser(ac->findUser(dbUser->email()));
                if (ac->user())
                    return;
            }
        }
        // download access control file from Dropbox
        loadAccessControl();
        QString dbFile = cloudAccessControlFile();
        User *usr = ac->findUser(dbUser->email());
        if (usr) {
            ac->setUser(usr);
        } else {
            QString email = dbUser->email();
            sql_class *sql = &Singleton<sql_class>::Instance();
            int congregationId = sql->getSetting("congregation_id").toInt();
            cpersons cp;
            Person *p = cp.getPersonByEmail(email, congregationId);
            QString personUuid;
            if (p) {
                personUuid = p->uuid();
            }
            // create new access control file
            if (!dbFile.compare(defaultOnlineStorageDir + "access_control.json", Qt::CaseInsensitive)) {
                // the current user is owner of Dropbox folder
                // replace previously created admin user
                usr = ac->findUser("admin");
                if (usr) {
                    usr->setEmail(email);
                    usr->setPersonUuid(personUuid);
                    usr->setName(dbUser->name());
                    // make sure user has admin rights
                    if (adminRole)
                        usr->addRole(adminRole);
                }
            }
            if (!usr) {
                // create new user and give the viewer rights
                ac->addUser(dbUser->name(), email, personUuid);
                usr = ac->findUser(dbUser->email());
                if (publisherRole)
                    usr->addRole(publisherRole);
            }
            ac->setUser(usr);
        }
        // save cache file
        QJsonObject jsObj;
        ac->write(jsObj);
        QJsonDocument jsDoc(jsObj);
        if (localFile.open(QFile::WriteOnly))
            localFile.write(jsDoc.toJson());
    } else {
        // Not logged to Dropbox
        for (int i = ac->users().count(); i-- > 0;)
            ac->removeUser(ac->users()[i]->id());
        ac->addUser("ADMIN", "admin");
        User *usr = ac->findUser("admin");
        for (const Role &role : ac->roles()) {
            if (role.id() != Permission::RoleId::Administrator)
                usr->addRole(&role);
        }
        ac->setUser(usr);
    }
}

void cloud_controller::loadAccessControl()
{
    AccessControl *ac = &Singleton<AccessControl>::Instance();
    QSettings settings;

    QString acFile = cloudAccessControlFile();
    QByteArray b;
    QDateTime dt;
    storageService()->download(acFile, b, dt);
    if (dt.isValid()) {
        // load file
        bool valid = false;
        QJsonDocument tempJSDoc = QJsonDocument::fromJson(b);
        ac->load(tempJSDoc.object(), valid);
        settings.setValue("local_access_control", dt.toSecsSinceEpoch());
        // update local file
        QFile localFile(cachedAccessControlFile());
        if (localFile.open(QFile::WriteOnly))
            localFile.write(tempJSDoc.toJson());
    }
}

void cloud_controller::runTest()
{
    s.runTest();

    //    qDebug() << "runTest" << QThread::currentThread();
    //    QThread *thread = new QThread(this);
    //    sync_cloud *sc = new sync_cloud();

    //    sc->moveToThread(thread);
    //    connect(thread,&QThread::started,sc,&sync_cloud::runTest);
    //    thread->start();
}

bool cloud_controller::canResetCloudData()
{
    AccessControl *ac = &Singleton<AccessControl>::Instance();
    return isLoggedIn() && ac->user() && ac->user()->hasPermission(PermissionRule::CanDeleteCloudData);
}

void cloud_controller::resetCloudData()
{
    if (!canResetCloudData())
        return;
    emit cloudResetStarted();
    s.resetCloudData();
}

void cloud_controller::clearDatabase()
{
    sql_class *sql = &Singleton<sql_class>::Instance();
    sql->clearDatabase();
}

void cloud_controller::setStorageService(QString newValue, QString baseUrl)
{
    if (newValue.compare("Nextcloud", Qt::CaseInsensitive) == 0)
        setStorageService(new Nextcloud(baseUrl, this));
    else
        setStorageService(new Dropbox(constants::dropboxclientid(), this));
}

void cloud_controller::setStorageService(QPointer<StorageService> newValue)
{
    // clean up previous service
    if (storageService()) {
        QObject::disconnect(authorizeWithBrowserStartedConnection);
        storageService()->deleteLater();
    }
    // set new storage service
    b_storageService = newValue;
    authorizeWithBrowserStartedConnection = connect(storageService(), &StorageService::authorizeWithBrowserStarted, this, &cloud_controller::authorizeWithBrowserStarted);
    storageService()->loadAccount();
    s.setStorageService(storageService());
}

QString cloud_controller::debugBackground()
{
    QSettings settings;
    QString value = settings.value("mobile/backgroundtest", "").toString();
    return value;
}

void cloud_controller::setDebugBackground(QString value)
{
    QSettings settings;
    settings.setValue("mobile/backgroundtest", value);
}

void cloud_controller::syncTimeCheck()
{
    sql_class *sql = &Singleton<sql_class>::Instance();
    QDateTime dt;
    uint timestamp = QVariant(sql->getSetting("last_dbsync_time")).toUInt();
    if (timestamp > 0)
        dt.setSecsSinceEpoch(timestamp);
    setLastSyncTime(dt);
}

void cloud_controller::initStorageService()
{
    qDebug() << "Init storage service";
    storageService()->loadAccount();
    if (storageService()->account()->syncFile().isEmpty()) {
        // set default syncfile
        QString defaultSyncFile = defaultOnlineStorageDir + "syncfile.json";
        QDateTime dt = storageService()->getModifiedDate(defaultSyncFile);
        if (!dt.isValid())
            storageService()->upload(QByteArray("{}"), defaultSyncFile); // create an empty file
        storageService()->account()->setSyncFile(defaultSyncFile);
        // reset sync id
        sql_class *sql = &Singleton<sql_class>::Instance();
        sql->saveSetting("last_dbsync_id", "0");
        sql->saveSetting("last_dbsync_time", "-1");
    }
}

void cloud_controller::syncCloudConflict(int duplicateValues)
{
    auto ac = &Singleton<AccessControl>::Instance();
    Role adminRole;
    for (const Role &role : ac->roles()) {
        if (role.id() == Permission::RoleId::Administrator)
            adminRole = role;
    }
    auto user = ac->user();
    if (user->hasRole(adminRole)) {
        // Show message about local and cloud data conflict only for admins
        emit syncConflict(duplicateValues);
    } else {
        // Do not keep the local changes if user is not admin
        s.continueSync(false);
    }
}

void cloud_controller::syncCloudResetFound()
{
    qDebug() << "cloud reset found";
    auto ac = &Singleton<AccessControl>::Instance();
    Role adminRole;
    for (const Role &role : ac->roles()) {
        if (role.id() == Permission::RoleId::Administrator)
            adminRole = role;
    }
    auto user = ac->user();

    if (user->hasRole(adminRole)) {
        // Show message about the cloud reset only for admins
        emit cloudResetFound();
    } else {
        // Clear database
        clearDatabase();
        // Start new synchronization
        synchronize();
    }
}

const QString cloud_controller::cachedAccessControlFile()
{
    QString dirPath = QStandardPaths::writableLocation(QStandardPaths::AppDataLocation);
    if (!QDir(dirPath).exists())
        QDir().mkdir(dirPath);
    return dirPath + QDir::separator() + "access_control.json";
}

const QString cloud_controller::cloudAccessControlFile()
{
    if (!isLoggedIn())
        return "";
    const QFileInfo syncFileInfo(storageService()->account()->syncFile());
    QString dropboxFolder = syncFileInfo.path();
    return dropboxFolder + "/access_control.json";
}

void cloud_controller::databaseChanged(const QString tablename)
{
    if (tablename != "e_reminder" && tablename != "lmm_workbookregex") {
        SyncState newstate = (mSyncState == SyncState::Download) ? SyncState::Both : SyncState::Upload;
        if (newstate != mSyncState) {
            mSyncState = newstate;
            sql_class *sql = &Singleton<sql_class>::Instance();
            sql->saveSetting("local_changes", "true");
            emit stateChanged(mSyncState);
        }
    }
}
