/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2018, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef CCONGREGATION_H
#define CCONGREGATION_H

#include <QString>
#include <QList>
#include <QPair>
#include <QAbstractListModel>
#include <QSortFilterProxyModel>
#include <QValidator>
#include "assignmentInfo.h"
#include "sql_class.h"
#include "specialevent.h"

class MeetingDayAndTime : public QObject
{
    Q_OBJECT
    Q_PROPERTY(int id READ id CONSTANT FINAL)
    Q_PROPERTY(int ofYear READ ofYear WRITE setOfYear BINDABLE bindableOfYear)
    Q_PROPERTY(int meetingDay READ meetingDay WRITE setMeetingDay BINDABLE bindableMeetingDay)
    Q_PROPERTY(QString meetingTime READ meetingTime WRITE setMeetingTime BINDABLE bindableMeetingTime)

public:
    MeetingDayAndTime(QObject *parent = nullptr);
    MeetingDayAndTime(int id, int ofYear, int meetingDay, QString meetingTime, QObject *parent = nullptr);

    int id() const;

    int ofYear() { return b_ofYear.value(); }
    void setOfYear(int newValue) { b_ofYear = newValue; }
    QBindable<int> bindableOfYear() { return &b_ofYear; }

    int meetingDay() { return b_meetingDay.value(); }
    void setMeetingDay(int newValue) { b_meetingDay = newValue; }
    QBindable<int> bindableMeetingDay() { return &b_meetingDay; }

    QString meetingTime() { return b_meetingTime.value(); }
    void setMeetingTime(QString newValue) { b_meetingTime = newValue; }
    QBindable<QString> bindableMeetingTime() { return &b_meetingTime; }

    bool isValid() { return b_isValid.value(); }
    bool isDirty() { return b_isDirty.value(); }

    bool save();

signals:
    void ofYearChanged();
    void meetingDayChanged();
    void meetingTimeChanged();
    void isDirtyChanged();

private:
    int m_id;
    Q_OBJECT_BINDABLE_PROPERTY(MeetingDayAndTime, int, b_ofYear, &MeetingDayAndTime::ofYearChanged);
    Q_OBJECT_BINDABLE_PROPERTY(MeetingDayAndTime, int, b_meetingDay, &MeetingDayAndTime::meetingDayChanged);
    Q_OBJECT_BINDABLE_PROPERTY(MeetingDayAndTime, QString, b_meetingTime, &MeetingDayAndTime::meetingTimeChanged);
    QProperty<bool> b_isValid { false };
    Q_OBJECT_BINDABLE_PROPERTY(MeetingDayAndTime, bool, b_isDirty, &MeetingDayAndTime::isDirtyChanged);
};

class ccongregation : public QObject
{
    Q_OBJECT

public:
    class congregation;

    class meeting_dayandtime
    {
    public:
        meeting_dayandtime(); // should only be used in the one "blank" object in congregation

        int getId() const;
        void setId(int value);

        int getOfyear() const;
        void setOfyear(int value);

        int getMeetingday() const;
        void setMeetingday(int value);

        QString getMeetingtime() const;
        void setMeetingtime(const QString &value);

        /**
         * @brief ccongregation::meeting_dayandtime::getMeetingDate - translates weekof to the actual date of the talk
         * @param weekOf
         * @return date of the talk
         */
        QDate getMeetingDate(QDate weekOf) const;

        bool getIsvalid() const;
        void setIsvalid(bool value);

        bool getIsdirty() const;
        void setIsdirty(bool value);

    private:
        int id;
        bool isdirty;
        int ofyear;
        int meetingday;
        QString meetingtime;
    };

    class congregation
    {
    public:
        congregation();

        // returns true if the congregation is not 'None' (so one that comes from the Database)
        bool isValid() const;

        int id;
        QString name;
        QString time_meeting1;
        QString address;
        QString circuit;
        QString info;
        bool save();

        meeting_dayandtime const &getPublicmeeting(QDate onDate) const;
        meeting_dayandtime const &getPublicmeeting(int onYear) const;
        meeting_dayandtime const &getPublicmeeting_now() const;
        meeting_dayandtime const &getPublicmeeting_next() const;

        meeting_dayandtime &getPublicmeeting(QDate onDate);
        meeting_dayandtime &getPublicmeeting(int onYear);
        meeting_dayandtime &getPublicmeeting_now();
        meeting_dayandtime &getPublicmeeting_next();

    private:
        meeting_dayandtime blank;
        meeting_dayandtime publicmeeting_now;
        meeting_dayandtime publicmeeting_next;
    };

    enum meetings {
        tms,
        sm,
        pm,
        wt
    };
    Q_ENUM(meetings)

    ccongregation(QObject *parent = nullptr);

    /**
     * @brief getAllCongregations - retrieves all congregations' information except meeting day/times
     *                            - so far, no code that calls this makes use of it
     *                            - no need to pull it from the database unless it's used
     * @param groupedByCircuit
     * @return list of congregations
     */
    QList<congregation> getAllCongregations(bool groupedByCircuit = false);

    int getCongregationId(QString name);
    ccongregation::congregation getOrAddCongregation(QString name);
    ccongregation::congregation addCongregation(QString name);
    bool removeCongregation(int id);

    QString getCongregationName(int id);
    ccongregation::congregation getCongregationById(int id);
    int getCongregationIDByPartialName(QString name);
    ccongregation::congregation getMyCongregation();

    /**
     * @brief getSpecialEventId - Get id of the special event in the specific week
     * @param date - The first day of week
     * @return - SpecialEvent-Id
     */
    Q_INVOKABLE int getSpecialEventId(QDate date);

    /**
     * @brief getSpecialEventRule - Get rule for the special event of the specific week
     * @param date - The first day of week
     * @return - SpecialEventRule
     */
    SpecialEventRule *getSpecialEventRule(QDate date); // better not to allow QML to access the object, but rather let it find it in its own SpecialEvents-Instance

    /**
     * @brief noMeeting - is there a meeting on 'date'
     * @param date - The first day of week
     * @return - true if no midweek or/and weekend meeting for that week
     */
    Q_INVOKABLE bool noMeeting(QDate date);

    /**
     * @brief getExceptionText - Get exception text (ex. 'Circuit overseer's visit')
     * @param date - First day of week
     * @param addNoMeetingText - Add "(No meeting)"-text if there's no regular meeting
     * @return - Exception text
     */
    Q_INVOKABLE QString getExceptionText(QDate date, bool useDefaultIfEmpty, bool addNoMeetingText = false);

    /**
     * @brief getExceptionDates - Get exception date range
     * @param weekDate - First day of week
     * @param date1 - start date of exception
     * @param date2 - end date of exception
     * @return - true if exception found
     */
    bool getExceptionDates(const QDate weekDate, QDate &date1, QDate &date2);

    /**
     * @brief getMeetingDay - Get day of meeting. Deprecated; use getMeetingDay(MeetingType meetingType, QDate date)
     * @param date - First date of week
     * @param meetingtype - meetingtypes are cbs, tms, sm, pm and wt
     * @return - Day of week
     */
    Q_INVOKABLE int getMeetingDay(QDate date, meetings meetingtype);

    /**
     * @brief getMeetingDay - Get day of meeting
     * @param MeetingType - type of meeting
     * @param date - First date of week
     * @return - Day of week according to special event rules; if date is omitted, default meeting day is returned
     */
    Q_INVOKABLE int getMeetingDay(MeetingType meetingType, QDate date = QDate());

    /**
     * @brief setMidweekMeetingDay - Set day of midweek meeting
     *                               Monday = 1, Tuesday = 2 etc.
     * @param value - Day number
     */
    void setMidweekMeetingDay(int value);

    /**
     * @brief clearExceptionCache - clear memorized exceptions. Useful when database table has changed
     */
    Q_INVOKABLE void clearExceptionCache();

private:
    sql_class *sql;
    SpecialEvents *se;
    QHash<QDate, sql_item> cachedExceptions;
    QDate minCachedException;
    void updateExCache(QDate date);
};

class Congregation : public QObject
{
    Q_OBJECT
    Q_PROPERTY(int congregationId READ congregationId BINDABLE bindableCongregationId)
    Q_PROPERTY(QString congregationName READ congregationName WRITE setCongregationName BINDABLE bindableCongregationName)
    Q_PROPERTY(QString midweekMeetingTime READ midweekMeetingTime WRITE setMidweekMeetingTime BINDABLE bindableMidweekMeetingTime)
    Q_PROPERTY(QString address READ address WRITE setAddress BINDABLE bindableAddress)
    Q_PROPERTY(QString circuit READ circuit WRITE setCircuit BINDABLE bindableCircuit)
    Q_PROPERTY(QString info READ info WRITE setInfo BINDABLE bindableInfo)
    Q_PROPERTY(MeetingDayAndTime *currentPublicMeetingDayAndTime READ currentPublicMeetingDayAndTime WRITE setCurrentPublicMeetingDayAndTime BINDABLE bindableCurrentPublicMeetingDayAndTime)
    Q_PROPERTY(MeetingDayAndTime *nextPublicMeetingDayAndTime READ nextPublicMeetingDayAndTime WRITE setNextPublicMeetingDayAndTime BINDABLE bindableNextPublicMeetingDayAndTime)
    Q_PROPERTY(bool isLocalCongregation READ isLocalCongregation BINDABLE bindableIsLocalCongregation)
    Q_PROPERTY(bool isValid READ isValid BINDABLE bindableIsValid)

public:
    Congregation(QObject *parent = nullptr);
    Congregation(int id, QString congregationName, QString midweekMeetingTime, QString address, QString circuit, QString info, bool isLocalCongregation = false, QObject *parent = nullptr);
    Congregation(int id, QString congregationName, QString midweekMeetingTime, QString address, QString circuit, QString info, MeetingDayAndTime *currentPublicMeetingDayAndTime, MeetingDayAndTime *nextPublicMeetingDayAndTime, bool isLocalCongregation, QObject *parent = nullptr);
    ~Congregation();

    int congregationId() { return b_congregationId.value(); }
    void setCongregationId(int newValue) { b_congregationId = newValue; }
    QBindable<int> bindableCongregationId() { return &b_congregationId; }

    QString congregationName() { return b_congregationName.value(); }
    void setCongregationName(QString newValue) { b_congregationName = newValue; }
    QBindable<QString> bindableCongregationName() { return &b_congregationName; }

    QString midweekMeetingTime() { return b_midweekMeetingTime.value(); }
    void setMidweekMeetingTime(QString newValue) { b_midweekMeetingTime = newValue; }
    QBindable<QString> bindableMidweekMeetingTime() { return &b_midweekMeetingTime; }

    QString address() { return b_address.value(); }
    void setAddress(QString newValue) { b_address = newValue; }
    QBindable<QString> bindableAddress() { return &b_address; }

    QString circuit() { return b_circuit.value(); }
    void setCircuit(QString newValue) { b_circuit = newValue; }
    QBindable<QString> bindableCircuit() { return &b_circuit; }

    QString info() { return b_info.value(); }
    void setInfo(QString newValue) { b_info = newValue; }
    QBindable<QString> bindableInfo() { return &b_info; }

    QPointer<MeetingDayAndTime> currentPublicMeetingDayAndTime() { return b_currentPublicMeetingDayAndTime.value(); }
    void setCurrentPublicMeetingDayAndTime(QPointer<MeetingDayAndTime> newValue) { b_currentPublicMeetingDayAndTime = newValue; }
    QBindable<QPointer<MeetingDayAndTime>> bindableCurrentPublicMeetingDayAndTime() { return &b_currentPublicMeetingDayAndTime; }

    QPointer<MeetingDayAndTime> nextPublicMeetingDayAndTime() { return b_nextPublicMeetingDayAndTime.value(); }
    void setNextPublicMeetingDayAndTime(QPointer<MeetingDayAndTime> newValue) { b_nextPublicMeetingDayAndTime = newValue; }
    QBindable<QPointer<MeetingDayAndTime>> bindableNextPublicMeetingDayAndTime() { return &b_nextPublicMeetingDayAndTime; }

    MeetingDayAndTime *publicMeetingDayAndTime(QDate onDate);
    MeetingDayAndTime *publicMeetingDayAndTime(int onYear);

    bool isLocalCongregation() const { return b_isLocalCongregation.value(); }
    QBindable<bool> bindableIsLocalCongregation() { return &b_isLocalCongregation; }

    bool isDirty() { return b_isDirty.value(); }

    bool isValid() const { return b_isValid.value(); }
    QBindable<bool> bindableIsValid() { return &b_isValid; }

    Q_INVOKABLE bool save();

signals:
    void congregationIdChanged();
    void congregationNameChanged();
    void midweekMeetingTimeChanged();
    void addressChanged();
    void circuitChanged();
    void infoChanged();
    void currentPublicMeetingDayAndTimeChanged();
    void nextPublicMeetingDayAndTimeChanged();

private:
    Q_OBJECT_BINDABLE_PROPERTY(Congregation, int, b_congregationId, &Congregation::congregationIdChanged);
    Q_OBJECT_BINDABLE_PROPERTY(Congregation, QString, b_congregationName, &Congregation::congregationNameChanged);
    Q_OBJECT_BINDABLE_PROPERTY(Congregation, QString, b_midweekMeetingTime, &Congregation::midweekMeetingTimeChanged);
    Q_OBJECT_BINDABLE_PROPERTY(Congregation, QString, b_address, &Congregation::addressChanged);
    Q_OBJECT_BINDABLE_PROPERTY(Congregation, QString, b_circuit, &Congregation::circuitChanged);
    Q_OBJECT_BINDABLE_PROPERTY(Congregation, QString, b_info, &Congregation::infoChanged);
    QProperty<QPointer<MeetingDayAndTime>> b_currentPublicMeetingDayAndTime;
    QProperty<QPointer<MeetingDayAndTime>> b_nextPublicMeetingDayAndTime;
    QProperty<bool> b_isLocalCongregation { false };
    QProperty<bool> b_isDirty { false };
    QProperty<bool> b_isValid { false };
};

class CongregationTreeItem : public QObject
{
    Q_OBJECT

public:
    explicit CongregationTreeItem(const QList<QVariant> &data, CongregationTreeItem *parentItem = nullptr);
    ~CongregationTreeItem();

    CongregationTreeItem *child(int row);
    Q_INVOKABLE int childCount() const;
    int columnCount() const;
    Q_INVOKABLE QVariant data(int column) const;
    Q_INVOKABLE QVariant value(int column) const;
    int row() const;
    void appendChild(CongregationTreeItem *child);
    bool insertChildren(int position, int count);
    bool removeChildren(int position, int count);
    CongregationTreeItem *parentItem();
    bool setData(int column, const QVariant &value);

private:
    QList<CongregationTreeItem *> m_childItems;
    QList<QVariant> m_itemData;
    CongregationTreeItem *m_parentItem;
};

class CongregationTreeModel : public QAbstractItemModel
{
    Q_OBJECT
    Q_PROPERTY(QList<QString> circuits READ circuits NOTIFY circuitsChanged FINAL)

public:
    enum NodeTypes {
        CongregationNode = Qt::UserRole,
        PersonNode
    };
    Q_ENUM(NodeTypes)
    enum Roles {
        None = 0,
        TitleRole = Qt::UserRole,
        TypeRole,
        CongregationIdRole,
        CongregationRole,
        PersonIdRole,
        PersonRole
    };
    Q_ENUM(Roles)

    explicit CongregationTreeModel(QObject *parent = nullptr);
    ~CongregationTreeModel();

    QHash<int, QByteArray> roleNames() const override;

    QVariant data(const QModelIndex &index, int role) const override;
    Qt::ItemFlags flags(const QModelIndex &index) const override;
    bool setData(const QModelIndex &index, const QVariant &value,
                 int role = Qt::EditRole) override;
    Q_INVOKABLE QVariantMap get(int row) const;
    Q_INVOKABLE QModelIndex index(int row, int column,
                                  const QModelIndex &parent = QModelIndex()) const override;

    Q_INVOKABLE QModelIndex getCongregationIndex(int congregationId) const;
    Q_INVOKABLE QModelIndex getMyCongregationIndex() const;
    Q_INVOKABLE QModelIndex getPersonIndex(int personId) const;
    Q_INVOKABLE QModelIndex parent(const QModelIndex &index) const override;

    int rowCount(const QModelIndex &parent = QModelIndex()) const override;
    int columnCount(const QModelIndex &parent = QModelIndex()) const override;

    bool insertRows(int position, int rows,
                    const QModelIndex &parent = QModelIndex()) override;

    bool removeRows(int position, int rows,
                    const QModelIndex &parent = QModelIndex()) override;

    Q_INVOKABLE void updateModel();
    Q_INVOKABLE void save();

    Q_INVOKABLE int addPerson(int congregationId);
    Q_INVOKABLE void removePerson(int personId);
    Q_INVOKABLE QVariant findPerson(int personId);
    Q_INVOKABLE int addCongregation();
    Q_INVOKABLE void removeCongregation(int congregationId);
    QModelIndex getGroupNodeIndex(QVariant value) const;

    QList<QString> circuits() const { return m_circuits; }
    bool isDuplicate(const QVariant &newValue, const QModelIndex &searchItemIndex = QModelIndex());
    Q_INVOKABLE bool changeCongregation(int personId, int newCongregationId);

signals:
    void notification();
    void personRemoved();
    void congregationRemoved();
    void modelChanged();
    void circuitsChanged();

private:
    sql_class *sql;
    void setupModelData(CongregationTreeItem *parent);
    CongregationTreeItem *getItem(const QModelIndex &index) const;
    CongregationTreeItem *rootItem;
    QList<QString> m_circuits;

    void updateCircuits();
};

class CongregationTreeSFProxyModel : public QSortFilterProxyModel
{
    Q_OBJECT
    Q_PROPERTY(QObject *source READ source WRITE setSource NOTIFY sourceChanged)
    Q_PROPERTY(QModelIndex filterRootIndex READ filterRootIndex WRITE setFilterRootIndex NOTIFY filterRootIndexChanged FINAL)
    Q_PROPERTY(bool filterByCircuit READ filterByCircuit WRITE setFilterByCircuit NOTIFY filterByCircuitChanged FINAL)
    Q_PROPERTY(QString circuit READ circuit WRITE setCircuit NOTIFY circuitChanged FINAL)
    Q_PROPERTY(QString filterText READ filterText WRITE setFilterText NOTIFY filterTextChanged FINAL)
    Q_PROPERTY(QByteArray sortRole READ sortRole WRITE setSortRole NOTIFY sortRoleChanged FINAL)
    Q_PROPERTY(Qt::SortOrder sortOrder READ sortOrder WRITE setSortOrder NOTIFY sortOrderChanged FINAL)

public:
    explicit CongregationTreeSFProxyModel(QObject *parent = nullptr);

    QObject *source() const;
    void setSource(QObject *source);

    QModelIndex filterRootIndex() const;
    void setFilterRootIndex(QModelIndex newValue);

    QString filterText() const;
    void setFilterText(QString newValue);

    bool filterByCircuit() const;
    void setFilterByCircuit(bool newValue);

    QString circuit() const;
    void setCircuit(QString newValue);

    QByteArray sortRole() const;
    void setSortRole(const QByteArray &role);

    void setSortOrder(Qt::SortOrder order);

    Q_INVOKABLE QVariantMap get(int row);

    Q_INVOKABLE virtual void sort(int column, Qt::SortOrder order = Qt::AscendingOrder) override
    {
        QSortFilterProxyModel::sort(column, order);
    }

protected:
    bool filterAcceptsRow(int sourceRow, const QModelIndex &sourceParent) const override;
    bool lessThan(const QModelIndex &left, const QModelIndex &right) const override;
    int roleKey(const QByteArray &role) const;

signals:
    void sourceChanged();
    void filterRootIndexChanged();
    void filterByCircuitChanged();
    void circuitChanged();
    void filterTextChanged();
    void sortRoleChanged();
    void sortOrderChanged();

private:
    QModelIndex m_filterRootIndex;
    bool m_filterByCircuit;
    QString m_circuit;
    QString m_filterText;
};

class CongregationValidator : public QValidator
{
    Q_OBJECT
    Q_PROPERTY(CongregationTreeModel *model READ model WRITE setModel NOTIFY modelChanged)
    Q_PROPERTY(Congregation *congregation READ congregation WRITE setCongregation NOTIFY congregationChanged)
    Q_PROPERTY(Person *person READ person WRITE setPerson NOTIFY personChanged)
    Q_PROPERTY(CongregationValidator::Field field READ field WRITE setField NOTIFY fieldChanged)

public:
    enum Field {
        None = 0,
        CongregationName = Qt::UserRole,
        FirstName,
        LastName,
        FullName
    };
    Q_ENUM(Field)
    CongregationValidator(QObject *parent = nullptr);
    ~CongregationValidator();

    QValidator::State validate(QString &input, int &pos) const override;

    CongregationTreeModel *model() const;
    void setModel(CongregationTreeModel *newModel);
    Congregation *congregation() const;
    void setCongregation(Congregation *newCongregation);
    Person *person() const;
    void setPerson(Person *newPerson);
    Field field() const;
    void setField(Field newField);

Q_SIGNALS:
    void modelChanged();
    void congregationChanged();
    void personChanged();
    void fieldChanged();
    void errorChanged(const QString &error) const;

private:
    CongregationTreeModel *m_model;
    Congregation *m_congregation;
    Person *m_person;
    Field m_field;
};
#endif // CCONGREGATION_H
