/**
 * This file is part of TheocBase.
 *
 * Copyright (C) 2011-2018, TheocBase Development Team, see AUTHORS.
 *
 * TheocBase is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * TheocBase is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TheocBase.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "ccongregation.h"
#include "qforeach.h"
#include <ciso646>
#include "accesscontrol.h"
#include "todo.h"

MeetingDayAndTime::MeetingDayAndTime(QObject *parent)
    : MeetingDayAndTime(-1, 0, 7, "", parent)
{
}

MeetingDayAndTime::MeetingDayAndTime(int id, int ofYear, int meetingDay, QString meetingTime, QObject *parent)
    : QObject(parent), m_id(id), b_ofYear(ofYear), b_meetingDay(meetingDay), b_meetingTime(meetingTime)
{
    QObject::connect(this, &MeetingDayAndTime::ofYearChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &MeetingDayAndTime::meetingDayChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &MeetingDayAndTime::meetingTimeChanged, [&]() { b_isDirty = true; });

    b_isValid.setBinding([&]() {
        return this->ofYear() > 2000 && this->ofYear() < 9999 && this->meetingDay() > 0 && this->meetingDay() < 8 && this->meetingTime().contains(":");
    });
}

int MeetingDayAndTime::id() const
{
    return m_id;
}

bool MeetingDayAndTime::save()
{
    if (not this->isValid())
        return false;

    AccessControl *ac = &Singleton<AccessControl>::Instance();
    if (!ac->user() || !ac->user()->hasPermission(PermissionRule::CanEditCongregationSettings))
        return false;

    // save changes to database
    if (!isDirty())
        return true;

    sql_class *sql = &Singleton<sql_class>::Instance();
    sql_item s;
    s.insert("mtg_day", meetingDay());
    s.insert("mtg_time", meetingTime());
    bool savedOK = sql->updateSql("congregationmeetingtimes", "id", QVariant(id()).toString(), &s);
    if (!savedOK)
        return savedOK;
    b_isDirty = false;
    return savedOK;
}

ccongregation::ccongregation(QObject *parent)
    : QObject(parent)
{
    sql = &Singleton<sql_class>::Instance();
    se = &SpecialEvents::Instance();
    clearExceptionCache();
}

ccongregation::congregation ccongregation::getMyCongregation()
{
    return getCongregationById(sql->getSetting("congregation_id").toInt());
}

QList<ccongregation::congregation> ccongregation::getAllCongregations(bool groupedByCircuit)
{
    // return all congregations <id,name>

    QList<ccongregation::congregation> congregations;

    sql_items cgr;
    QString query = "SELECT * FROM congregations WHERE active ORDER BY ";
    if (groupedByCircuit) {
        query.append("circuit, name");
    } else {
        query.append("name");
    }
    cgr = sql->selectSql(query);
    if (!cgr.empty()) {
        for (unsigned int i = 0; i < cgr.size(); i++) {
            sql_item s = cgr[i];
            congregation c;
            c.id = s.value("id").toInt();
            c.name = s.value("name").toString();
            c.time_meeting1 = s.value("meeting1_time").toString();
            c.address = s.value("address").toString();
            c.circuit = s.value("circuit").toString();
            c.info = s.value("info").toString();
            congregations.append(c);
        }
    } else {
        // create default
        congregation c = addCongregation("-");
        sql->saveSetting("congregation_id", QString::number(c.id));
        congregations.append(c);
    }
    return congregations;
}

int ccongregation::getCongregationId(QString name)
{
    // get congregation id by name

    sql_item values;
    values.insert(":name", name);
    return sql->selectScalar("select id from congregations where name = :name and active", &values, -1).toInt();
}

ccongregation::congregation ccongregation::getOrAddCongregation(QString name)
{
    int id = getCongregationId(name);
    if (id < 1)
        return addCongregation(name);
    else
        return getCongregationById(id);
}

ccongregation::congregation ccongregation::addCongregation(QString name)
{
    // add new congregation
    sql_item newsrk;
    newsrk.insert("name", name);
    int srkid = sql->insertSql("congregations", &newsrk, "id");
    sql->updateThisYearMeetingTimes();
    sql->updateNextYearMeetingTimes();
    // calling getCongregationId will fill meeting time objects properly
    return getCongregationById(srkid);
}

bool ccongregation::removeCongregation(int id)
{
    sql_item s;
    s.insert("id", id);
    s.insert("time_stamp", 0);
    return sql->execSql("update congregations set active = 0, time_stamp = :time_stamp where id = :id", &s, true);
    // return sql->removeSql("congregations","id = " + QVariant(id).toString());
}

QString ccongregation::getCongregationName(int id)
{
    sql_items s = sql->selectSql("congregations", "id", QVariant(id).toString(), "");
    return s[0].value("name").toString();
}

ccongregation::congregation ccongregation::getCongregationById(int id)
{
    sql_items s = sql->selectSql("congregations", "id", QVariant(id).toString(), "");
    congregation c;
    if (s.empty()) {
        c.name = tr("(Missing Record)", "database is now missing this entry");
    } else {
        sql_item citem = s[0];
        c.name = citem.value("name").toString();
        c.id = citem.value("id").toInt();
        c.address = citem.value("address").toString();
        c.time_meeting1 = citem.value("meeting1_time").toString();
        c.circuit = citem.value("circuit").toString();
        c.info = citem.value("info").toString();

        int thisyear = QDate::currentDate().year();
        sql_item criteria;
        criteria.insert(":congregation_id", id);
        criteria.insert(":thisyear", thisyear);
        s = sql->selectSql("select id, mtg_year, mtg_day, mtg_time from congregationmeetingtimes where congregation_id = :congregation_id and (mtg_year = :thisyear or mtg_year = :thisyear + 1)", &criteria);
        for (sql_items::size_type i = 0; i != s.size(); i++) {
            sql_item mt = s[i];
            int yr = mt["mtg_year"].toInt();
            if ((yr == thisyear) || (yr == thisyear + 1)) {
                auto &mtgtime = (yr == thisyear)
                        ? c.getPublicmeeting_now()
                        : c.getPublicmeeting_next();

                mtgtime.setId(mt["id"].toInt());
                mtgtime.setOfyear(mt["mtg_year"].toInt());
                mtgtime.setMeetingday(mt["mtg_day"].toInt());
                mtgtime.setMeetingtime(mt["mtg_time"].toString());
                mtgtime.setIsdirty(false);
            }
        }
    }
    return c;
}

int ccongregation::getCongregationIDByPartialName(QString name)
{
    QVariant id = sql->selectScalar("select id from congregations where name like '%" + sql->EscapeQuotes(name) + "%' and active", nullptr, 0);
    return id.toInt();
}

ccongregation::meeting_dayandtime::meeting_dayandtime()
    : id(-1), isdirty(false), ofyear(0), meetingday(7), meetingtime("")
{
}

int ccongregation::meeting_dayandtime::getId() const
{
    return id;
}

void ccongregation::meeting_dayandtime::setId(int value)
{
    id = value;
}

QString ccongregation::meeting_dayandtime::getMeetingtime() const
{
    return meetingtime;
}

void ccongregation::meeting_dayandtime::setMeetingtime(const QString &value)
{
    if (meetingtime.compare(value)) {
        meetingtime = value;
        isdirty = true;
    }
}

int ccongregation::meeting_dayandtime::getMeetingday() const
{
    return meetingday;
}

void ccongregation::meeting_dayandtime::setMeetingday(int value)
{
    if (meetingday != value) {
        meetingday = value;
        isdirty = true;
    }
}

/**
 * @brief ccongregation::meeting_dayandtime::getMeetingDate - translates weekof to the actual date of the talk
 * @param weekOf
 * @return date of the talk
 */
QDate ccongregation::meeting_dayandtime::getMeetingDate(QDate weekOf) const
{
    return weekOf.addDays(meetingday - 1);
}

int ccongregation::meeting_dayandtime::getOfyear() const
{
    return ofyear;
}

void ccongregation::meeting_dayandtime::setOfyear(int value)
{
    ofyear = value;
}

bool ccongregation::meeting_dayandtime::getIsvalid() const
{
    return ofyear > 2000 && ofyear < 9999 && meetingday > 0 && meetingday < 8 && meetingtime.contains(":");
}

bool ccongregation::meeting_dayandtime::getIsdirty() const
{
    return isdirty;
}

void ccongregation::meeting_dayandtime::setIsdirty(bool value)
{
    isdirty = value;
}

// init
ccongregation::congregation::congregation()
    : id(-1), name(""), time_meeting1(""), address(""), circuit(""), info(""), blank(), publicmeeting_now(), publicmeeting_next()
{
    int thisyear = QDate::currentDate().year();
    publicmeeting_now.setOfyear(thisyear++);
    publicmeeting_next.setOfyear(thisyear);
}

bool ccongregation::congregation::isValid() const
{
    // this test comes from orignal ccongregation::congregation::save() code
    // QUESTION: why not id > 0 ??? is id==0 a valid one ?
    return id >= 0;
}

int ccongregation::getSpecialEventId(QDate date)
{
    int id = -1;
    updateExCache(date);
    if (cachedExceptions.contains(date)) {
        sql_item ex(cachedExceptions[date]);
        id = ex.value("type").toInt();
    }
    return id;
}

SpecialEventRule *ccongregation::getSpecialEventRule(QDate date)
{
    int id = getSpecialEventId(date);
    return se->findSpecialEventRule(id);
}

bool ccongregation::noMeeting(QDate date)
{
    updateExCache(date);

    if (cachedExceptions.contains(date)) {
        sql_item ex(cachedExceptions[date]);
        SpecialEventRule *specialEventRule = se->findSpecialEventRule(ex.value("type").toInt());
        bool isNoMeeting(specialEventRule->exclusivity() == SpecialEventExclusivity::NoOtherMeetingInSameWeek);

        if (!isNoMeeting) {
            int midweekday = ex.value("schoolday").toInt();
            int weekendday = ex.value("publicmeetingday").toInt();
            isNoMeeting = (midweekday == 0 || weekendday == 0);
        }
        return isNoMeeting;
    } else {
        return false;
    }
}

QString ccongregation::getExceptionText(QDate date, bool useDefaultIfEmpty, bool addNoMeetingText)
{
    updateExCache(date);
    sql_item v = cachedExceptions.value(date);
    if (!v.empty()) {
        SpecialEventRule *specialEventRule = se->findSpecialEventRule(v.value("type").toInt());
        QString description(specialEventRule->canChangeDescription() ? v.value("desc").toString() : specialEventRule->description());
        if (useDefaultIfEmpty && description.isEmpty())
            description = specialEventRule->description();
        if (addNoMeetingText) {
            bool isNoMmeeting((specialEventRule->exclusivity() == SpecialEventExclusivity::NoOtherMeetingInSameWeek) || ((v.value("schoolday").toInt() + v.value("publicmeetingday").toInt()) == 0));
            if (isNoMmeeting)
                return QObject::tr("%1 (No meeting)", "no meeting exception type").arg(description);
        }
        return description;
    } else {
        return "";
    }
}

bool ccongregation::getExceptionDates(const QDate weekDate, QDate &date1, QDate &date2)
{
    sql_item v = cachedExceptions.value(weekDate);
    if (v.empty()) {
        return false;
    } else {
        date1 = v.value("date").toDate();
        date2 = v.value("date2").toDate();
        return true;
    }
}

int ccongregation::getMeetingDay(QDate date, meetings meetingtype)
{
    // return meeting day
    MeetingType meetingType;
    switch (meetingtype) {
    case tms:
    case sm:
        meetingType = MeetingType::MidweekMeeting;
        break;
    default:
        meetingType = MeetingType::WeekendMeeting;
        break;
    }
    return getMeetingDay(meetingType, date);
}

int ccongregation::getMeetingDay(MeetingType meetingType, QDate date)
{
    // return meeting day
    int ret;

    // get default meeting day
    if (meetingType == MeetingType::MidweekMeeting) {
        ret = sql->getIntSetting("school_day", 1);
    } else {
        ret = getMyCongregation().getPublicmeeting(date.isValid() ? date : QDate::currentDate()).getMeetingday();
    }

    if (!date.isValid())
        return ret;

    updateExCache(date);
    sql_item v = cachedExceptions.value(date);

    if (!v.empty()) {
        SpecialEventRule *specialEventRule = se->findSpecialEventRule(v.value("type").toInt());
        int schoolday = v.value("schoolday").toInt();
        int publicmeetingday = v.value("publicmeetingday").toInt();
        // int cbsday = v.value("cbsday").toInt();
        if (specialEventRule->isCircuitOverseersVisit()) {
            // circuit overseer's visit
            if (meetingType == MeetingType::MidweekMeeting) {
                // life and ministry meeting
                ret = 2;
            } else {
                // use default value for public meeting
                ret = getMyCongregation().getPublicmeeting(date).getMeetingday();
            }
        } else if (specialEventRule->exclusivity() == SpecialEventExclusivity::NoOtherMeetingInSameWeek) {
            // convention -> no meetings
            ret = 0;
        } else if (specialEventRule->exclusivity() == SpecialEventExclusivity::NoOtherMeetingInSamePartOfTheWeek) {
            // memorial
            // when the memorial falls on a weekday, no midweek meeting will be scheduled
            // when the memorial falls on a weekend, no weekend meeting will be scheduled
            if (v.value("date").toDate().dayOfWeek() <= 5)
                ret = meetingType == MeetingType::MidweekMeeting ? 0 : publicmeetingday;
            else
                ret = meetingType == MeetingType::MidweekMeeting ? schoolday : 0;
        } else {
            // other exception
            if (meetingType == MeetingType::MidweekMeeting) {
                ret = schoolday;
            } else {
                ret = publicmeetingday;
            }
        }
    }
    return ret;
}

void ccongregation::setMidweekMeetingDay(int value)
{
    // Save value to settings table in the database
    sql->saveSetting("school_day", QVariant(value).toString());
}

void ccongregation::clearExceptionCache()
{
    cachedExceptions.clear();
    minCachedException = QDate::fromString("12-31-2035", "MM-dd-yyyy");
}

void ccongregation::updateExCache(QDate date)
{
    if (date < minCachedException) {
        sql_items s;
        s = sql->selectSql(QString("SELECT * FROM exceptions where date <= '%1' AND date2 >= '%2' and active").arg(minCachedException.toString(Qt::ISODate), date.toString(Qt::ISODate)));
        for (sql_item v : s) {
            QDate date1 = v.value("date").toDate();
            date1 = date1.addDays(-date1.dayOfWeek() + 1);
            QDate date2 = v.value("date2").toDate();
            date2 = date2.addDays(7 - date2.dayOfWeek());
            for (QDate d = date1; d <= date2; d = d.addDays(1)) {
                cachedExceptions.insert(d, v);
            }
        }
        minCachedException = date;
    }
}

bool ccongregation::congregation::save()
{
    if (not this->isValid())
        return false;

    auto sql = &Singleton<sql_class>::Instance();

    sql_item s;
    s.insert("address", this->address);
    s.insert("meeting1_time", time_meeting1);
    s.insert("name", this->name);
    s.insert("circuit", this->circuit);
    s.insert("info", this->info);
    bool savedOK = sql->updateSql("congregations", "id", QVariant(this->id).toString(), &s);
    if (!savedOK)
        return savedOK;

    ccongregation::meeting_dayandtime const &mtgtime_now = this->getPublicmeeting_now();
    if (mtgtime_now.getIsdirty() && mtgtime_now.getIsvalid()) {
        s.clear();
        s.insert("mtg_day", mtgtime_now.getMeetingday());
        s.insert("mtg_time", mtgtime_now.getMeetingtime());
        savedOK = sql->updateSql("congregationmeetingtimes", "id", QVariant(mtgtime_now.getId()).toString(), &s);
    }
    ccongregation::meeting_dayandtime &mtgtime = this->getPublicmeeting_next();
    if (!mtgtime.getIsvalid()) {
        mtgtime.setMeetingday(this->getPublicmeeting_now().getMeetingday());
        mtgtime.setMeetingtime(this->getPublicmeeting_now().getMeetingtime());
    }
    if (mtgtime.getIsdirty() && mtgtime.getIsvalid()) {
        s.clear();
        s.insert("mtg_day", mtgtime.getMeetingday());
        s.insert("mtg_time", mtgtime.getMeetingtime());
        savedOK = sql->updateSql("congregationmeetingtimes", "id", QVariant(mtgtime.getId()).toString(), &s);
    }
    return savedOK;
}

ccongregation::meeting_dayandtime const &ccongregation::congregation::getPublicmeeting(QDate onDate) const
{
    ccongregation::meeting_dayandtime const &dayandtime = this->getPublicmeeting(onDate.year());
    QDate ptDate(onDate.addDays(dayandtime.getMeetingday() - 1));
    // call it again to make sure we have the time for the actual meeting day, not week of
    return this->getPublicmeeting(ptDate.year());
}

ccongregation::meeting_dayandtime const &ccongregation::congregation::getPublicmeeting(int onYear) const
{
    if (onYear > publicmeeting_now.getOfyear())
        return publicmeeting_next;
    return publicmeeting_now;
}

ccongregation::meeting_dayandtime const &ccongregation::congregation::getPublicmeeting_next() const
{
    return publicmeeting_next;
}

ccongregation::meeting_dayandtime const &ccongregation::congregation::getPublicmeeting_now() const
{
    return publicmeeting_now;
}

ccongregation::meeting_dayandtime &ccongregation::congregation::getPublicmeeting(QDate onDate)
{
    ccongregation::meeting_dayandtime const &dayandtime = this->getPublicmeeting(onDate.year());
    QDate ptDate(onDate.addDays(dayandtime.getMeetingday() - 1));
    // call it again to make sure we have the time for the actual meeting day, not week of
    return this->getPublicmeeting(ptDate.year());
}

ccongregation::meeting_dayandtime &ccongregation::congregation::getPublicmeeting(int onYear)
{
    if (onYear > publicmeeting_now.getOfyear())
        return publicmeeting_next;
    return publicmeeting_now;
}

ccongregation::meeting_dayandtime &ccongregation::congregation::getPublicmeeting_next()
{
    return publicmeeting_next;
}

ccongregation::meeting_dayandtime &ccongregation::congregation::getPublicmeeting_now()
{
    return publicmeeting_now;
}

Congregation::Congregation(QObject *parent)
    : Congregation(-1, "", "", "", "", "", false, parent)
{
}

Congregation::Congregation(int id, QString congregationName, QString midweekMeetingTime, QString address, QString circuit, QString info, bool isLocalCongregation, QObject *parent)
    : Congregation(id, congregationName, midweekMeetingTime, address, circuit, info, new MeetingDayAndTime(), new MeetingDayAndTime(), isLocalCongregation, parent)
{
}

Congregation::Congregation(int id, QString congregationName, QString midweekMeetingTime, QString address, QString circuit, QString info, MeetingDayAndTime *currentPublicMeetingDayAndTime, MeetingDayAndTime *nextPublicMeetingDayAndTime, bool isLocalCongregation, QObject *parent)
    : QObject(parent), b_congregationId(id), b_congregationName(congregationName), b_midweekMeetingTime(midweekMeetingTime), b_address(address), b_circuit(circuit), b_info(info), b_currentPublicMeetingDayAndTime(currentPublicMeetingDayAndTime), b_nextPublicMeetingDayAndTime(nextPublicMeetingDayAndTime), b_isLocalCongregation(isLocalCongregation)
{
    int thisYear = QDate::currentDate().year();
    if (this->currentPublicMeetingDayAndTime().isNull())
        setCurrentPublicMeetingDayAndTime(new MeetingDayAndTime());
    if (this->nextPublicMeetingDayAndTime().isNull())
        setNextPublicMeetingDayAndTime(new MeetingDayAndTime());
    this->currentPublicMeetingDayAndTime()->setOfYear(thisYear++);
    this->nextPublicMeetingDayAndTime()->setOfYear(thisYear);

    QObject::connect(this, &Congregation::congregationNameChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Congregation::midweekMeetingTimeChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Congregation::addressChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Congregation::circuitChanged, [&]() { b_isDirty = true; });
    QObject::connect(this, &Congregation::infoChanged, [&]() { b_isDirty = true; });

    b_isValid.setBinding([&]() {
        // check validity of a congregation
        return congregationId() > 0 && !this->congregationName().isEmpty();
    });
}

Congregation::~Congregation()
{
}

MeetingDayAndTime *Congregation::publicMeetingDayAndTime(QDate onDate)
{
    MeetingDayAndTime *dayAndTime = this->publicMeetingDayAndTime(onDate.year());
    QDate ptDate(onDate.addDays(dayAndTime->meetingDay() - 1));
    // call it again to make sure we have the time for the actual meeting day, not week of
    return this->publicMeetingDayAndTime(ptDate.year());
}

MeetingDayAndTime *Congregation::publicMeetingDayAndTime(int onYear)
{
    if (onYear > currentPublicMeetingDayAndTime()->ofYear())
        return b_nextPublicMeetingDayAndTime.value();
    return currentPublicMeetingDayAndTime();
}

bool Congregation::save()
{
    if (not this->isValid())
        return false;

    AccessControl *ac = &Singleton<AccessControl>::Instance();
    if (!ac->user() || !ac->user()->hasPermission(PermissionRule::CanEditCongregationSettings))
        return false;

    // save changes to database
    if (!isDirty()
        && !currentPublicMeetingDayAndTime()->isDirty()
        && !nextPublicMeetingDayAndTime()->isDirty())
        return true;

    sql_class *sql = &Singleton<sql_class>::Instance();
    sql_item s;
    s.insert("address", this->address());
    s.insert("meeting1_time", this->midweekMeetingTime());
    s.insert("name", this->congregationName());
    s.insert("circuit", this->circuit());
    s.insert("info", this->info());
    bool savedOK = sql->updateSql("congregations", "id", QVariant(congregationId()).toString(), &s);
    if (!savedOK)
        return savedOK;

    b_isDirty = false;

    // save meeting times
    savedOK = currentPublicMeetingDayAndTime()->save();
    if (!savedOK)
        return savedOK;
    MeetingDayAndTime *nextMeetingTime = this->nextPublicMeetingDayAndTime();
    if (!nextMeetingTime->isValid()) {
        nextMeetingTime->setMeetingDay(this->currentPublicMeetingDayAndTime()->meetingDay());
        nextMeetingTime->setMeetingTime(this->currentPublicMeetingDayAndTime()->meetingTime());
    }
    savedOK = nextPublicMeetingDayAndTime()->save();
    if (!savedOK)
        return savedOK;

    return savedOK;
}

CongregationTreeItem::CongregationTreeItem(const QList<QVariant> &data, CongregationTreeItem *parentItem)
    : m_itemData(data), m_parentItem(parentItem)
{
}

CongregationTreeItem::~CongregationTreeItem()
{
    qDeleteAll(m_childItems);
}

CongregationTreeItem *CongregationTreeItem::child(int row)
{
    if (row < 0 || row >= m_childItems.size())
        return nullptr;
    return m_childItems.at(row);
}

int CongregationTreeItem::childCount() const
{
    return m_childItems.count();
}

int CongregationTreeItem::columnCount() const
{
    return m_itemData.count();
}

QVariant CongregationTreeItem::data(int column) const
{
    if (column < 0 || column >= m_itemData.size())
        return QVariant();
    return m_itemData.at(column);
}

QVariant CongregationTreeItem::value(int column) const
{
    if (column < 0 || column >= m_itemData.size())
        return QVariant();
    return m_itemData.at(column);
}

void CongregationTreeItem::appendChild(CongregationTreeItem *item)
{
    m_childItems.append(item);
}

bool CongregationTreeItem::insertChildren(int position, int count)
{
    if (position < 0 || position > m_childItems.size())
        return false;

    for (int row = 0; row < count; ++row) {
        QList<QVariant> columnData;
        columnData << "";
        columnData << "";
        columnData << 0;
        columnData << 0;
        columnData << 0;
        columnData << 0;
        CongregationTreeItem *item = new CongregationTreeItem(columnData, this);
        m_childItems.insert(position, item);
    }

    return true;
}

bool CongregationTreeItem::removeChildren(int position, int count)
{
    if (position < 0 || position + count > m_childItems.size())
        return false;

    for (int row = 0; row < count; ++row)
        delete m_childItems.takeAt(position);

    return true;
}

CongregationTreeItem *CongregationTreeItem::parentItem()
{
    return m_parentItem;
}

int CongregationTreeItem::row() const
{
    if (m_parentItem)
        return m_parentItem->m_childItems.indexOf(const_cast<CongregationTreeItem *>(this));

    return 0;
}

bool CongregationTreeItem::setData(int column, const QVariant &value)
{
    if (column < 0 || column >= m_itemData.size())
        return false;

    m_itemData[column] = value;
    return true;
}

CongregationTreeModel::CongregationTreeModel(QObject *parent)
    : QAbstractItemModel(parent)
{
    sql = &Singleton<sql_class>::Instance();
    QList<QVariant> rootData;
    rootData << "Title"
             << "Type"
             << "Congregation-ID"
             << "Congregation"
             << "Person-ID"
             << "Person";
    rootItem = new CongregationTreeItem(rootData);
    updateModel();
}

CongregationTreeModel::~CongregationTreeModel()
{
    delete rootItem;
}

int CongregationTreeModel::columnCount(const QModelIndex &parent) const
{
    if (parent.isValid())
        return static_cast<CongregationTreeItem *>(parent.internalPointer())->columnCount();
    return rootItem->columnCount();
}

QVariant CongregationTreeModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid())
        return QVariant();

    CongregationTreeItem *item = static_cast<CongregationTreeItem *>(index.internalPointer());

    switch (role) {
    case TitleRole: {
        if (item->data(0).value<NodeTypes>() == CongregationNode) {
            Congregation *c = item->data(2).value<Congregation *>();
            if (c != nullptr)
                return c->congregationName();
            else
                return "";
        } else {
            Person *p = item->data(4).value<Person *>();
            if (p != nullptr)
                return p->fullName();
            else
                return "";
        }
        break;
    }
    // case SummaryRole:
    //     return item->data(1);
    case TypeRole:
        return item->data(0);
    // case CircuitIdRole:
    //     return item->data(2);
    // case CircuitRole:
    //     return item->data(4);
    case CongregationIdRole:
        return item->data(1);
    case CongregationRole:
        return item->data(2);
    case PersonIdRole:
        return item->data(3);
    case PersonRole:
        return item->data(4);
    default:
        return QVariant();
    }
    return item->data(index.column());
}

Qt::ItemFlags CongregationTreeModel::flags(const QModelIndex &index) const
{
    if (!index.isValid())
        return Qt::NoItemFlags;

    return QAbstractItemModel::flags(index);
}

bool CongregationTreeModel::setData(const QModelIndex &index, const QVariant &value, int role)
{
    CongregationTreeItem *item = getItem(index);

    bool result = false;
    switch (role) {
    case Qt::EditRole:
        result = item->setData(index.column(), value);
        break;
    // case TitleRole:
    //     result = item->setData(0, value);
    //     break;
    // case SummaryRole:
    //     result = item->setData(1, value);
    //     break;
    case TypeRole:
        result = item->setData(0, value);
        break;
    case CongregationIdRole:
        result = item->setData(1, value);
        break;
    case CongregationRole:
        result = item->setData(2, value);
        break;
    case PersonIdRole:
        result = item->setData(3, value);
        break;
    case PersonRole:
        result = item->setData(4, value);
        break;
    }

    if (result)
        emit dataChanged(index, index);

    return result;
}

QVariantMap CongregationTreeModel::get(int row) const
{
    QHash<int, QByteArray> names = roleNames();
    QHashIterator<int, QByteArray> i(names);
    QVariantMap res;
    QModelIndex idx = index(row, 0);
    while (i.hasNext()) {
        i.next();
        QVariant data = idx.data(i.key());
        res[i.value()] = data;
    }
    return res;
}

QModelIndex CongregationTreeModel::index(int row, int column, const QModelIndex &parent)
        const
{
    //    if (!hasIndex(row, column, parent))
    //        return QModelIndex();

    if (!rootItem || row < 0 || column < 0)
        return QModelIndex();

    CongregationTreeItem *parentItem;

    if (!parent.isValid())
        parentItem = rootItem;
    else
        parentItem = static_cast<CongregationTreeItem *>(parent.internalPointer());

    CongregationTreeItem *childItem = parentItem->child(row);
    if (childItem)
        return createIndex(row, column, childItem);
    return QModelIndex();
}

QModelIndex CongregationTreeModel::getCongregationIndex(int congregationId) const
{
    for (int congregationRow = 0; congregationRow < this->rowCount(); ++congregationRow) {
        QModelIndex congregationIndex = this->index(congregationRow, 0);

        if (congregationIndex.data(CongregationIdRole) == congregationId)
            return congregationIndex;
    }
    return QModelIndex();
}

QModelIndex CongregationTreeModel::getMyCongregationIndex() const
{
    return getCongregationIndex(sql->getSetting("congregation_id").toInt());
}

QModelIndex CongregationTreeModel::getPersonIndex(int personId) const
{
    for (int congregationRow = 0; congregationRow < this->rowCount(); ++congregationRow) {
        QModelIndex congregationIndex = this->index(congregationRow, 0);

        for (int personRow = 0; personRow < this->rowCount(congregationIndex); ++personRow) {
            QModelIndex personIndex = this->index(personRow, 0, congregationIndex);
            if (personIndex.data(PersonIdRole) == personId)
                return personIndex;
        }
    }
    return QModelIndex();
}

QModelIndex CongregationTreeModel::parent(const QModelIndex &index) const
{
    if (!index.isValid())
        return QModelIndex();

    CongregationTreeItem *childItem = static_cast<CongregationTreeItem *>(index.internalPointer());
    CongregationTreeItem *parentItem = childItem->parentItem();

    if (parentItem == rootItem)
        return QModelIndex();

    return createIndex(parentItem->row(), 0, parentItem);
}

int CongregationTreeModel::rowCount(const QModelIndex &parent) const
{
    CongregationTreeItem *parentItem;
    if (parent.column() > 0)
        return 0;

    if (!parent.isValid())
        parentItem = rootItem;
    else
        parentItem = static_cast<CongregationTreeItem *>(parent.internalPointer());

    return parentItem->childCount();
}

void CongregationTreeModel::setupModelData(CongregationTreeItem *parent)
{
    AccessControl *ac = &Singleton<AccessControl>::Instance();
    if (!ac->user()
        || !(ac->user()->hasPermission(PermissionRule::CanViewPublishers)
             || ac->user()->hasPermission(PermissionRule::CanViewPublicSpeakers)))
        return;

    CongregationTreeItem *rootItem = parent;
    CongregationTreeItem *currentParent = nullptr;

    int ownCongregationId = QVariant(sql->getSetting("congregation_id")).toInt();
    ccongregation cc;
    QList<ccongregation::congregation> congregationList;
    cpersons cp;
    QList<Person *> publicSpeakers = cp.getAllPersons(1);
    if (ac->user()->hasPermission(PermissionRule::CanViewPublicSpeakers)) {
        congregationList = cc.getAllCongregations(0);
        publicSpeakers = cp.getAllPersons(1);
    } else {
        congregationList.append(cc.getMyCongregation());
        publicSpeakers = cp.getAllPersons(2);
    }

    sql_items s;
    foreach (ccongregation::congregation c, congregationList) {
        int thisyear = QDate::currentDate().year();
        QPointer<MeetingDayAndTime> currentMeetingDayAndTime;
        QPointer<MeetingDayAndTime> nextMeetingDayAndTime;
        sql_item criteria;
        criteria.insert(":congregation_id", c.id);
        criteria.insert(":thisyear", thisyear);
        s = sql->selectSql("select id, mtg_year, mtg_day, mtg_time from congregationmeetingtimes where congregation_id = :congregation_id and (mtg_year = :thisyear or mtg_year = :thisyear + 1)", &criteria);
        for (sql_items::size_type i = 0; i != s.size(); i++) {
            sql_item mt = s[i];
            int yr = mt["mtg_year"].toInt();

            if (yr == thisyear) {
                currentMeetingDayAndTime = new MeetingDayAndTime(
                        mt["id"].toInt(),
                        mt["mtg_year"].toInt(),
                        mt["mtg_day"].toInt(),
                        mt["mtg_time"].toString(),
                        this);
            }
            if (yr == thisyear + 1) {
                nextMeetingDayAndTime = new MeetingDayAndTime(
                        mt["id"].toInt(),
                        mt["mtg_year"].toInt(),
                        mt["mtg_day"].toInt(),
                        mt["mtg_time"].toString(),
                        this);
            }
        }

        Congregation *congregation = new Congregation(c.id,
                                                      c.name,
                                                      c.time_meeting1,
                                                      c.address,
                                                      c.circuit,
                                                      c.info,
                                                      currentMeetingDayAndTime,
                                                      nextMeetingDayAndTime,
                                                      c.id == ownCongregationId,
                                                      this);
        QObject::connect(congregation, &Congregation::circuitChanged, this, &CongregationTreeModel::updateCircuits);

        // add congregation node
        QList<QVariant> columnData;
        columnData << CongregationNode; // type
        columnData << congregation->congregationId(); // congregation id
        columnData << QVariant::fromValue(congregation); // congregation
        columnData << 0; // person id
        columnData << 0; // person
        currentParent = new CongregationTreeItem(columnData, rootItem);
        rootItem->appendChild(currentParent);

        if (congregation->congregationId() == ownCongregationId) {
            QList<Person *> personList = cp.getAllPersons(0);
            for (Person *person : personList) {
                person->setParent(this);
                columnData.clear();
                columnData << PersonNode; // type
                columnData << congregation->congregationId(); // congregation id
                columnData << QVariant::fromValue(congregation); // congregation
                columnData << person->id(); // person id
                columnData << QVariant::fromValue(person); // person
                currentParent->appendChild(new CongregationTreeItem(columnData, currentParent));
            }
        } else {
            for (Person *speaker : publicSpeakers) {
                if (speaker->congregationId() != congregation->congregationId())
                    continue;
                speaker->setParent(this);
                columnData.clear();
                columnData << PersonNode; // type
                columnData << congregation->congregationId(); // congregation id
                columnData << QVariant::fromValue(congregation); // congregation
                columnData << speaker->id(); // person id
                columnData << QVariant::fromValue(speaker); // person
                currentParent->appendChild(new CongregationTreeItem(columnData, currentParent));
            }
        }
    }
}

void CongregationTreeModel::updateCircuits()
{
    m_circuits.clear();
    for (int congregationRow = 0; congregationRow < this->rowCount(); ++congregationRow) {
        QModelIndex congregationIndex = this->index(congregationRow, 0);
        Congregation *c = congregationIndex.data(CongregationRole).value<Congregation *>();
        // update circuit list
        if (!m_circuits.contains(c->circuit()))
            m_circuits.append(c->circuit());
    }
    // sort circuits
    std::sort(m_circuits.begin(), m_circuits.end(),
              [](QString a, QString b) {
                  return QString::compare(a, b, Qt::CaseInsensitive);
              });
    emit circuitsChanged();
}

bool CongregationTreeModel::isDuplicate(const QVariant &newValue, const QModelIndex &searchItemIndex)
{
    if (!searchItemIndex.isValid())
        return false;
    QModelIndex parent = searchItemIndex.parent();
    bool isDuplicate = false;
    for (int i = 0; i < this->rowCount(parent); ++i) {
        QModelIndex iIndex = this->index(i, 0, parent);
        if (searchItemIndex == iIndex)
            continue;
        QString iTitle = this->data(iIndex, CongregationTreeModel::TitleRole).toString();
        if (QString::compare(iTitle, newValue.toString(), Qt::CaseInsensitive) == 0) {
            isDuplicate = true;
            break;
        }
    }
    return isDuplicate;
}

bool CongregationTreeModel::changeCongregation(int personId, int newCongregationId)
{
    QModelIndex personIndex = getPersonIndex(personId);
    QModelIndex newCongregationIndex = getCongregationIndex(newCongregationId);
    if (personIndex.isValid() && newCongregationIndex.isValid()) {
        Person *person = personIndex.data(CongregationTreeModel::PersonRole).value<Person *>();
        if (person == nullptr || person->congregationId() == newCongregationId)
            return false;

        person->setCongregationId(newCongregationId);
        if (person->save()) {
            // move outgoing talks to To Do List
            todo::addScheduledTalks(personId, false);
            // reload data
            updateModel();
            return true;
        }
    }
    return false;
}

bool CongregationTreeModel::insertRows(int position, int rows, const QModelIndex &parent)
{
    CongregationTreeItem *parentItem = getItem(parent);
    bool success;

    beginInsertRows(parent, position, position + rows - 1);
    success = parentItem->insertChildren(position, rows);
    endInsertRows();

    return success;
}

bool CongregationTreeModel::removeRows(int position, int rows,
                                       const QModelIndex &parent)
{
    CongregationTreeItem *parentItem = getItem(parent);
    bool success;

    if (position < 0 || rows < 1 || position + rows > parentItem->childCount())
        return false;

    beginRemoveRows(parent, position, position + rows - 1);
    success = parentItem->removeChildren(position, rows);
    endRemoveRows();

    return success;
}

QHash<int, QByteArray> CongregationTreeModel::roleNames() const
{
    auto roles = QAbstractItemModel::roleNames();
    roles[TitleRole] = "title";
    // roles[SummaryRole] = "summary";
    roles[TypeRole] = "type";
    // roles[CircuitIdRole] = "circuitId";
    // roles[CircuitRole] = "circuit";
    roles[CongregationIdRole] = "congregationId";
    roles[CongregationRole] = "congregation";
    roles[PersonIdRole] = "personId";
    roles[PersonRole] = "person";
    return roles;
}

void CongregationTreeModel::updateModel()
{
    beginResetModel();

    if (rootItem->childCount() > 0) {
        beginRemoveRows(QModelIndex(), 0, rootItem->childCount() - 1);
        rootItem->removeChildren(0, rootItem->childCount());
        endRemoveRows();
    }

    setupModelData(rootItem);

    endResetModel();
    emit modelChanged();
    updateCircuits();
}

void CongregationTreeModel::save()
{
    sql_class *sql = &Singleton<sql_class>::Instance();
    sql->startTransaction();
    for (int congregationRow = 0; congregationRow < this->rowCount(); ++congregationRow) {
        QModelIndex congregationIndex = this->index(congregationRow, 0);
        Congregation *congregation = congregationIndex.data(CongregationRole).value<Congregation *>();
        if (congregation != nullptr && congregation->isValid() && !isDuplicate(congregation->congregationName(), congregationIndex)) {
            // save congregation if valid
            congregation->save();
        }
        for (int personRow = 0; personRow < this->rowCount(congregationIndex); ++personRow) {
            QModelIndex personIndex = this->index(personRow, 0, congregationIndex);
            Person *person = personIndex.data(PersonRole).value<Person *>();
            if (person != nullptr && person->isValid() && !isDuplicate(person->fullName(), personIndex)) {
                // save person if valid
                person->save();
            }
        }
    }
    sql->commitTransaction();
}

CongregationTreeItem *CongregationTreeModel::getItem(const QModelIndex &index) const
{
    if (index.isValid()) {
        CongregationTreeItem *item = static_cast<CongregationTreeItem *>(index.internalPointer());
        if (item)
            return item;
    }
    return rootItem;
}

int CongregationTreeModel::addPerson(int congregationId)
{
    beginResetModel();
    // find congregation
    QModelIndex congregationIndex = getCongregationIndex(congregationId);
    if (congregationIndex.isValid()) {
        Congregation *congregation = this->data(congregationIndex, CongregationRole).value<Congregation *>();

        // insert new person row in congregation node
        int pos = this->rowCount(congregationIndex);
        if (this->insertRows(pos, 1, congregationIndex)) {
            QModelIndex newPersonIndex = this->index(pos, 0, congregationIndex);

            // add person in database
            Person *newPerson = new Person(this);
            newPerson->setGender(Person::Male);
            newPerson->setServant(false);
            newPerson->setLastName(tr("Last name"));
            newPerson->setFirstName(tr("First name"));
            newPerson->setCongregationId(congregationId);
            cpersons cp;
            int newPersonId = cp.addPerson(newPerson);
            newPerson->setId(newPersonId);

            // set person data
            this->setData(newPersonIndex, PersonNode, TypeRole);
            this->setData(newPersonIndex, congregation->congregationId(), CongregationIdRole);
            this->setData(newPersonIndex, QVariant::fromValue(congregation), CongregationRole);
            this->setData(newPersonIndex, newPersonId, PersonIdRole);
            this->setData(newPersonIndex, QVariant::fromValue(newPerson), PersonRole);

            return newPersonId;
        }
    }
    endResetModel();
    emit modelChanged();

    return 0;
}

void CongregationTreeModel::removePerson(int personId)
{
    cpersons *cp = new cpersons;
    if (cp->removePerson(personId)) {
        // update tree
        QModelIndex personIndex = getPersonIndex(personId);
        if (personIndex.isValid())
            if (removeRows(personIndex.row(), 1, personIndex.parent())) {
                emit personRemoved();
            }
    }
}

QVariant CongregationTreeModel::findPerson(int personId)
{
    QModelIndex personIndex = getPersonIndex(personId);
    if (personIndex.isValid()) {
        return personIndex.data(PersonRole);
    }
    return QVariant();
}

int CongregationTreeModel::addCongregation()
{
    beginResetModel();
    ccongregation cc;
    ccongregation::congregation c = cc.addCongregation(tr("New Congregation"));
    int thisyear = QDate::currentDate().year();

    QPointer<MeetingDayAndTime> currentMeetingDayAndTime;
    QPointer<MeetingDayAndTime> nextMeetingDayAndTime;
    sql_item criteria;
    criteria.insert(":congregation_id", c.id);
    criteria.insert(":thisyear", thisyear);
    sql_items s;
    s = sql->selectSql("select id, mtg_year, mtg_day, mtg_time from congregationmeetingtimes where congregation_id = :congregation_id and (mtg_year = :thisyear or mtg_year = :thisyear + 1)", &criteria);
    for (sql_items::size_type i = 0; i != s.size(); i++) {
        sql_item mt = s[i];
        int yr = mt["mtg_year"].toInt();

        if (yr == thisyear) {
            currentMeetingDayAndTime = new MeetingDayAndTime(
                    mt["id"].toInt(),
                    mt["mtg_year"].toInt(),
                    mt["mtg_day"].toInt(),
                    mt["mtg_time"].toString(),
                    this);
        }
        if (yr == thisyear + 1) {
            nextMeetingDayAndTime = new MeetingDayAndTime(
                    mt["id"].toInt(),
                    mt["mtg_year"].toInt(),
                    mt["mtg_day"].toInt(),
                    mt["mtg_time"].toString(),
                    this);
        }
    }

    int ownCongregationId = QVariant(sql->getSetting("congregation_id")).toInt();
    Congregation *congregation = new Congregation(c.id,
                                                  c.name,
                                                  c.time_meeting1,
                                                  c.address,
                                                  c.circuit,
                                                  c.info,
                                                  currentMeetingDayAndTime,
                                                  nextMeetingDayAndTime,
                                                  c.id == ownCongregationId,
                                                  this);
    QObject::connect(congregation, &Congregation::circuitChanged, this, &CongregationTreeModel::updateCircuits);

    // add congregation node
    QList<QVariant> columnData;
    columnData << CongregationNode; // type
    columnData << congregation->congregationId(); // congregation id
    columnData << QVariant::fromValue(congregation); // congregation
    columnData << 0; // person id
    columnData << 0; // person
    CongregationTreeItem *congregationItem = new CongregationTreeItem(columnData, rootItem);
    rootItem->appendChild(congregationItem);

    endResetModel();
    emit modelChanged();
    updateCircuits();

    return congregation->congregationId();
}

void CongregationTreeModel::removeCongregation(int congregationId)
{
    QModelIndex congregationIndex = getCongregationIndex(congregationId);
    if (congregationIndex.isValid()) {
        // remove persons
        for (int personRow = this->rowCount(congregationIndex) - 1; personRow >= 0; --personRow) {
            QModelIndex personIndex = this->index(personRow, 0, congregationIndex);
            if (personIndex.isValid()) {
                int personId(personIndex.data(PersonIdRole).toInt());
                removePerson(personId);
            }
        }
        // remove congregation
        ccongregation *cc = new ccongregation;
        if (cc->removeCongregation(congregationId)) {
            // update tree
            if (removeRows(congregationIndex.row(), 1, congregationIndex.parent())) {
                emit congregationRemoved();
            }
        }
    }
}

QModelIndex CongregationTreeModel::getGroupNodeIndex(QVariant value) const
{
    int grpRows = this->rowCount();
    for (int i = 0; i < grpRows; i++) {
        QModelIndex idx = index(i, 0);

        if (this->data(idx, PersonIdRole) == value)
            return idx;
    }
    return QModelIndex();
}

CongregationTreeSFProxyModel::CongregationTreeSFProxyModel(QObject *parent)
    : QSortFilterProxyModel(parent), m_filterRootIndex(QModelIndex())
{
    setRecursiveFilteringEnabled(true);
}

QObject *CongregationTreeSFProxyModel::source() const
{
    return sourceModel();
}

void CongregationTreeSFProxyModel::setSource(QObject *source)
{
    setSourceModel(qobject_cast<QAbstractItemModel *>(source));
    emit sourceChanged();
}

QModelIndex CongregationTreeSFProxyModel::filterRootIndex() const
{
    return m_filterRootIndex;
}

void CongregationTreeSFProxyModel::setFilterRootIndex(QModelIndex newValue)
{
    beginFilterChange();
    m_filterRootIndex = newValue;
    endFilterChange(QSortFilterProxyModel::Direction::Rows);
    emit filterRootIndexChanged();
}

bool CongregationTreeSFProxyModel::filterByCircuit() const
{
    return m_filterByCircuit;
}

void CongregationTreeSFProxyModel::setFilterByCircuit(bool newValue)
{
    beginFilterChange();
    m_filterByCircuit = newValue;
    endFilterChange(QSortFilterProxyModel::Direction::Rows);
    emit filterByCircuitChanged();
}

QString CongregationTreeSFProxyModel::circuit() const
{
    return m_circuit;
}

void CongregationTreeSFProxyModel::setCircuit(QString newValue)
{
    beginFilterChange();
    m_circuit = newValue;
    endFilterChange(QSortFilterProxyModel::Direction::Rows);
    emit circuitChanged();
}

QString CongregationTreeSFProxyModel::filterText() const
{
    return m_filterText;
}

void CongregationTreeSFProxyModel::setFilterText(QString newValue)
{
    beginFilterChange();
    m_filterText = newValue;
    endFilterChange(QSortFilterProxyModel::Direction::Rows);
    emit filterTextChanged();
}

QByteArray CongregationTreeSFProxyModel::sortRole() const
{
    return roleNames().value(QSortFilterProxyModel::sortRole());
}

void CongregationTreeSFProxyModel::setSortRole(const QByteArray &role)
{
    QSortFilterProxyModel::setSortRole(roleKey(role));
    emit sortRoleChanged();
    QSortFilterProxyModel::invalidate();
}

void CongregationTreeSFProxyModel::setSortOrder(Qt::SortOrder order)
{
    QSortFilterProxyModel::sort(0, order);
    emit sortOrderChanged();
}

bool CongregationTreeSFProxyModel::filterAcceptsRow(int sourceRow,
                                                    const QModelIndex &sourceParent) const
{
    QModelIndex index = sourceModel()->index(sourceRow, 0, sourceParent);
    QModelIndex rootIndex = sourceModel()->index(filterRootIndex().row(), 0, filterRootIndex().parent());

    Person *p = sourceModel()->data(index, CongregationTreeModel::PersonRole).value<Person *>();
    Congregation *c = sourceModel()->data(index, CongregationTreeModel::CongregationRole).value<Congregation *>();
    QString title = sourceModel()->data(index, CongregationTreeModel::TitleRole).toString();

    bool result = (!filterByCircuit() || (c != nullptr && QString::compare(c->circuit(), circuit(), Qt::CaseInsensitive) == 0))
            && ((rootIndex.isValid() && index == rootIndex)
                || (p != nullptr && (p->fullName().contains(filterText(), Qt::CaseInsensitive)))
                || title.contains(filterText(), Qt::CaseInsensitive));
    return result;
}

QVariantMap CongregationTreeSFProxyModel::get(int row)
{
    QHash<int, QByteArray> names = roleNames();
    QHashIterator<int, QByteArray> i(names);
    QVariantMap res;
    QModelIndex idx = index(row, 0);
    while (i.hasNext()) {
        i.next();
        QVariant data = idx.data(i.key());
        res[i.value()] = data;
    }
    return res;
}

int CongregationTreeSFProxyModel::roleKey(const QByteArray &role) const
{
    QHash<int, QByteArray> roles = roleNames();
    QHashIterator<int, QByteArray> it(roles);
    while (it.hasNext()) {
        it.next();
        if (it.value() == role)
            return it.key();
    }
    return -1;
}

bool CongregationTreeSFProxyModel::lessThan(const QModelIndex &left, const QModelIndex &right) const
{
    QVariant id = sourceModel()->data(left, CongregationTreeModel::PersonIdRole);
    if (id.toInt() == 0) {
        // group node; keep sort order from the model
    } else {
        switch (roleKey(sortRole())) {
        case CongregationTreeModel::CongregationRole: {
            QVariant val1 = sourceModel()->data(left, roleKey(sortRole()));
            QVariant val2 = sourceModel()->data(right, roleKey(sortRole()));
            if (!val1.isNull() && !val2.isNull()) {
                return val1.toInt() < val2.toInt();
            }
            break;
        }
        case CongregationTreeModel::PersonRole: {
            QVariant val1 = sourceModel()->data(left, roleKey(sortRole()));
            QVariant val2 = sourceModel()->data(right, roleKey(sortRole()));
            int compResult = QString::compare(val1.toString(), val2.toString(), Qt::CaseInsensitive);
            return compResult < 0;
            break;
        }
        default:
            break;
        }
    }
    return false;
}

CongregationValidator::CongregationValidator(QObject *parent)
    : QValidator(parent), m_model(nullptr), m_congregation(nullptr), m_person(nullptr), m_field(CongregationValidator::Field::None)
{
}

CongregationValidator::~CongregationValidator()
{
}

QValidator::State CongregationValidator::validate(QString &input, int &pos) const
{
    Q_UNUSED(pos)

    if (!model()) {
        emit errorChanged("");
        return Acceptable;
    }
    if (!congregation()) {
        emit errorChanged("");
        return Acceptable;
    }
    if (field() == CongregationValidator::Field::None) {
        emit errorChanged("");
        return Acceptable;
    }

    // add validation here...
    QModelIndex congregationIndex = model()->getCongregationIndex(congregation()->congregationId());
    if (field() == CongregationValidator::Field::CongregationName) {
        // congregation name
        if (input.isEmpty()) {
            emit errorChanged(tr("The congregation name is a mandatory field."));
            return Intermediate;
        }
        // check if the congregation was added multiple times
        if (model()->isDuplicate(input, congregationIndex)) {
            emit errorChanged(tr("A congregation with the same name exists already."));
            return Intermediate;
        }
    }
    if (!person()) {
        emit errorChanged("");
        return Acceptable;
    }
    QModelIndex personIndex = model()->getPersonIndex(person()->id());
    if (field() == CongregationValidator::Field::FullName) {
        // check if a person with the same name exists already (within the same congregation)
        if (model()->isDuplicate(input, personIndex)) {
            emit errorChanged(tr("A person with the same name exists already."));
            return Intermediate;
        }
    }
    if (field() == CongregationValidator::Field::FirstName) {
        // first name
        if (input.isEmpty()) {
            emit errorChanged(tr("The first name is a mandatory field."));
            return Intermediate;
        }
    }
    if (field() == CongregationValidator::Field::LastName) {
        // last name
        if (input.isEmpty()) {
            emit errorChanged(tr("The last name is a mandatory field."));
            return Intermediate;
        }
    }
    if (field() == CongregationValidator::Field::FirstName
        || field() == CongregationValidator::Field::LastName) {
        // check if a person with the same name exists already (within the same congregation)
        bool isDuplicate = false;
        QModelIndex congregationIndex = model()->getCongregationIndex(congregation()->congregationId());
        for (int i = 0; i < model()->rowCount(congregationIndex); ++i) {
            QModelIndex iPersonIndex = model()->index(i, 0, congregationIndex);
            Person *iPerson = model()->data(iPersonIndex, CongregationTreeModel::PersonRole).value<Person *>();
            if (iPerson != nullptr) {
                QString firstName = field() == CongregationValidator::Field::FirstName
                        ? input
                        : person()->firstName();
                QString lastName = field() == CongregationValidator::Field::LastName
                        ? input
                        : person()->lastName();

                if (QString::compare(iPerson->firstName(), firstName, Qt::CaseInsensitive) == 0
                    && QString::compare(iPerson->lastName(), lastName, Qt::CaseInsensitive) == 0
                    && iPerson->congregationId() == person()->congregationId()
                    && iPerson->id() != person()->id()) {
                    isDuplicate = true;
                    break;
                }
            }
        }
        if (isDuplicate) {
            emit errorChanged(tr("A person with the same name exists already."));
            return Intermediate;
        }
    }

    emit errorChanged("");
    return Acceptable;
}

CongregationTreeModel *CongregationValidator::model() const
{
    return m_model;
}

void CongregationValidator::setModel(CongregationTreeModel *newModel)
{
    if (m_model == newModel)
        return;
    m_model = newModel;
    emit modelChanged();
}

Congregation *CongregationValidator::congregation() const
{
    return m_congregation;
}

void CongregationValidator::setCongregation(Congregation *newCongregation)
{
    if (m_congregation == newCongregation)
        return;
    m_congregation = newCongregation;
    emit congregationChanged();
}

Person *CongregationValidator::person() const
{
    return m_person;
}

void CongregationValidator::setPerson(Person *newPerson)
{
    if (m_person == newPerson)
        return;
    m_person = newPerson;
    emit personChanged();
}

CongregationValidator::Field CongregationValidator::field() const
{
    return m_field;
}

void CongregationValidator::setField(CongregationValidator::Field newField)
{
    if (m_field == newField)
        return;
    m_field = newField;
    emit fieldChanged();
}
